<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class ParseCreateTable
{
    /**
     * An instance
     *
     * @var \InSegment\ApiCore\Services\ParseCreateTable
     */
    private static $instance = null;

    /**
     * An array holding show creates for tables
     *
     * @var array
     */
    private $showCreateData;

    /**
     * Prefix of service data in cache
     *
     * @var string
     */
    private $cachePrefix;

    /**
     * Lists column unique data of loaded table
     *
     * @param string $table
     * @return string[][] - list of lists of columns of unique keys
     */
    public function getColumnUniqueData(string $table)
    {
        if (!isset($this->showCreateData['columnUnique'][$table])) {
            $this->loadColumnData($table);
        }

        return $this->showCreateData['columnUnique'][$table];
    }

    /**
     * Lists primary keys of loaded table
     *
     * @param string $table
     * @return array | [
     *     'columns' => string[] - list of columns the primary key contains
     *     'isAi' => bool        - is PK an Auto-increment one
     * ] | null                  - null if the table has no PK
     */
    public function getPrimaryKeyData(string $table)
    {
        if (!isset($this->showCreateData['primaryKeys'][$table])) {
            $this->loadColumnData($table);
        }

        return $this->showCreateData['primaryKeys'][$table];
    }

    /**
     * Returns list of constraints of loaded table.
     *
     * @param string $table Name of table.
     * @return array | [
     *     'raw' => string            - raw statement from `SHOW CREATE TABLE`
     *     'name' => string           - name of constraint
     *     'isForeignKey' => string   - true if constraint if `FOREIGN KEY`
     * ] | null                       - null if table has no constraints
     */
    public function getConstraintsData(string $table)
    {
        if (!isset($this->showCreateData['constraints'][$table])) {
            $this->loadColumnData($table);
        }

        return $this->showCreateData['constraints'][$table];
    }

    /**
     * Returns statement for create full copy of the table.
     *
     * @param string $table Name of table.
     * @return string
     */
    public function getCreateTableStatement(string $table)
    {
        if (!isset($this->showCreateData['showCreate'][$table])) {
            $this->loadColumnData($table);
        }

        return implode("\n", $this->showCreateData['showCreate'][$table]);
    }

    /**
     * Get auto_increment PRIMARY key
     *
     * @param string $table
     * @return string|null
     */
    public function getAutoKey(string $table)
    {
        $pk = $this->getPrimaryKeyData($table);

        if ($pk !== null && $pk['isAi']) {
            return $pk['columns'][0];
        }

        return null;
    }

    /**
     * Get query to create temporary table
     *
     * If it is insert table, AUTO_INCREMENT key will be replaced with bigint(20) suitable for UUID_SHORT()
     * CONSTRAINT's are removed from query
     *
     * @param string $table
     * @param array $uuidableTableMentions
     * @param bool $uuidable
     * @param bool $allowInMemory
     * @param bool $allBools
     * @return array
     */
    public function getCreateForTemp(string $table, array $uuidableTableMentions, bool $uuidable, bool $allowInMemory, bool $allBools)
    {
        $this->loadShowCreate($table);

        $showCreate = $this->showCreateData['showCreate'][$table];
        $hasMentions = false;
        $last = count($showCreate) - 1;
        $pkData = $this->getPrimaryKeyData($table);
        
        if ($pkData === null || count($pkData['columns']) > 1) {
            $pk = null;
        } else {
            $pk = $pkData['columns'][0];
        }

        foreach ($showCreate as $i => $part) {
            $trimmed = trim($part);

            if ($trimmed[0] === '`') {
                if ($uuidableTableMentions && ($keyName = $this->mentionsMatches($trimmed, $uuidableTableMentions))) {
                    $hasMentions = true;
                    $showCreate[$i] = "`{$keyName}` bigint(20) unsigned NULL,";
                } else if ($allowInMemory && !$allBools && preg_match('/` ((tiny|medium|long)?(text|blob)|json)[ ,\(]/i', $trimmed)) {
                    $allowInMemory = false;
                }
                
                if ($allBools && ($keyName = $this->getColumnName($trimmed)) && $keyName !== $pk) {
                    $showCreate[$i] = "`{$keyName}` tinyint(1) unsigned NOT NULL default 0,";
                }
            } else if (($allBools && strpos($trimmed, 'UNIQUE KEY') === 0) || Str::startsWith($trimmed, 'CONSTRAINT')) {
                unset($showCreate[$i]);

                if ($last == $i + 1) {
                    for ($j = $i; $j > 0; --$j) {
                        if (isset($showCreate[$j])) {
                            $showCreate[$j] = rtrim($showCreate[$j], ',');
                            break;
                        }
                    }
                }
            }
        }

        if ($allowInMemory) {
            $showCreate[$last] = str_replace('ENGINE=InnoDB', 'ENGINE=MEMORY', $showCreate[$last]);
        }

        if ($uuidable && $hasMentions) {
            $showCreate[$last] = preg_replace('/AUTO_INCREMENT=\d+/', '', $showCreate[$last], 1);
        }

        return $showCreate;
    }

    /**
     * Sets the data about the primary key and unique key columns of the table
     *
     * @param string $table
     * @return $this
     */
    public function loadColumnData(string $table)
    {
        if (!isset($this->showCreateData['columnUnique'][$table])) {
            $this->loadShowCreate($table);
            $cacheKey = $this->getColumnUniqueCacheKey($table);

            // TODO: add flag to disable caching
            $cachedColumnData = Cache::rememberForever($cacheKey, function () use ($table) {
                $columnUniqueForTable = [];
                $primaryKeyInfo = null;
                $constraints = [];
                $showCreate = $this->showCreateData['showCreate'][$table];
                $createTable = array_shift($showCreate);
                $engineAIAndCharset = array_pop($showCreate);
                $aiColumnName = null;

                foreach ($showCreate as $tableStatement) {
                    $tableStatement = trim($tableStatement);
                    $primary = strpos($tableStatement, 'PRIMARY KEY') === 0;
                    $isConstraint = strpos($tableStatement, 'CONSTRAINT') === 0;
                    $unique = !$primary && strpos($tableStatement, 'UNIQUE KEY') === 0;

                    if ($tableStatement[0] === '`' && strpos($tableStatement, 'AUTO_INCREMENT') > -1) {
                        $aiColumnName = substr($tableStatement, 1, strpos($tableStatement, '`', 1) - 1);
                    }

                    if ($primary || $unique) {
                        $matches = [];
                        if (preg_match('/\(`([^\)]+)`\)/', $tableStatement, $matches)) {
                            $matchedColumnName = explode('`,`', $matches[1]);
                            $columnUniqueForTable[] = $matchedColumnName;
                            if ($primary) {
                                $primaryKeyInfo = [
                                    'columns' => $matchedColumnName,
                                    'isAi' => in_array($aiColumnName, $matchedColumnName)
                                ];
                            }
                        }
                    } elseif ($isConstraint) {
                        // TODO: encapsulate into separate `Constraint`-class
                        $constraint = [
                            'raw' => $tableStatement,
                            'name' => null,
                            'columns' => null,
                            'referenceTableName' => null,
                            'referenceColumns' => [],
                            'isForeignKey' => false,
                            'onUpdate' => null,
                            'onDelete' => null,
                        ];

                        // example of constraint-string for regexps below:
                        // CONSTRAINT `manual_verifications_record_id_foreign` FOREIGN KEY (`record_id`) REFERENCES `user_records` (`id`) ON DELETE NO ACTION ON UPDATE CASCADE
                        // TODO: should these regexps will combine into a single one ?

                        if (preg_match('/^CONSTRAINT `([^`]+)`/', $tableStatement, $matches)) {
                            $constraint['name'] = $matches[1];
                        }

                        if (preg_match('/\(`([^\)]+)`\) REFERENCES/', $tableStatement, $matches)) {
                            $constraint['columns'] = explode('`,`', $matches[1]);
                        }

                        if (preg_match('/REFERENCES `([^\`]+)` \(`([^\)]+)`\)/', $tableStatement, $matches)) {
                            $constraint['referenceTableName'] = $matches[1];
                            $constraint['referenceColumns'] = explode('`,`', $matches[2]);
                        }

                        if (preg_match('/ON DELETE (RESTRICT|CASCADE|SET NULL|NO ACTION|SET DEFAULT)/', $tableStatement, $matches)) {
                            $constraint['onDelete'] = $matches[1];
                        }

                        if (preg_match('/ON UPDATE (RESTRICT|CASCADE|SET NULL|NO ACTION|SET DEFAULT)/', $tableStatement, $matches)) {
                            $constraint['onUpdate'] = $matches[1];
                        }

                        $constraint['isForeignKey'] = preg_match('/FOREIGN KEY/', $tableStatement, $matches) !== 0;

                        $constraints[] = $constraint;
                    }
                }

                return ['primary' => $primaryKeyInfo, 'unique' => $columnUniqueForTable, 'constraints' => $constraints];
            });

            $this->showCreateData['primaryKeys'][$table] = $cachedColumnData['primary'];
            $this->showCreateData['columnUnique'][$table] = $cachedColumnData['unique'];
            $this->showCreateData['constraints'][$table] = $cachedColumnData['constraints'];
        }

        return $this;
    }

    /**
     * Sets the data about table creation specifics
     *
     * @param string $table
     * @return $this
     */
    protected function loadShowCreate(string $table)
    {
        if (!isset($this->showCreateData['showCreate'][$table])) {
            $showCreateKey = $this->getShowCreateCacheKey($table);
            $this->showCreateData['showCreate'][$table] = Cache::rememberForever($showCreateKey, function () use ($table) {
                return explode("\n", DB::selectOne("SHOW CREATE TABLE `{$table}`")->{'Create Table'});
            });
        }

        return $this;
    }

    /**
     * Test for uuidable mentions match, remove matched from further matching
     *
     * @param string $trimmed
     * @param array $mentions
     * @return bool|string
     */
    protected function mentionsMatches(string $trimmed, array &$mentions)
    {
        $matches = [];
        if (preg_match('/^`(' . implode('|', $mentions) . ')`/', $trimmed, $matches)) {
            unset($mentions[$matches[1]]);
            return $matches[1];
        }

        return false;
    }
    
    /**
     * Get column name from trimmed string
     * 
     * @param string $trimmed
     * @return boolean|array
     */
    protected function getColumnName(string $trimmed)
    {
        $matches = [];
        if (preg_match('/^`([^`]+)`/', $trimmed, $matches)) {
            return $matches[1];
        }
        
        return false;
    }

    /**
     * Get ParseCreateTable instance
     *
     * @return \InSegment\ApiCore\Services\ParseCreateTable
     */
    public static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new ParseCreateTable();
        }

        return self::$instance;
    }

    /**
     * A singleton, no instantiation from outside
     */
    private function __construct()
    {
        $this->showCreateData = [
            'primaryKeys' => [],
            'columnUnique' => [],
            'showCreate' => [],
            'constraints' => [],
        ];

        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'show-create-data';
    }



    // ----------------
    // RELATED TO CACHE
    // ----------------

    /**
     * Clears cached data for specific table.
     * Maybe it will be cool to add ability to accept no arguments and in this case clear whole cached data related
     * to ParseCreateTable but it's require support of wildcard in `Cache::forget` method.
     *
     * @param string $tableName Name of table for which cached data would be erased.
     */
    public function clearCache(string $tableName): void {
        $cacheKeyColumnUnique = $this->getColumnUniqueCacheKey($tableName);
        Cache::forget($cacheKeyColumnUnique);

        $cacheKeyShowCreate = $this->getShowCreateCacheKey($tableName);
        Cache::forget($cacheKeyShowCreate);

        unset(
            $this->showCreateData['primaryKeys'][$tableName],
            $this->showCreateData['columnUnique'][$tableName],
            $this->showCreateData['showCreate'][$tableName],
            $this->showCreateData['constraints'][$tableName]
        );
    }

    /**
     * Returns cache-key for caching data related to unique columns of $tableName table.
     *
     * @param string $tableName Name of table to cache,
     * @return string
     */
    private function getColumnUniqueCacheKey(string $tableName): string {
        return "{$this->cachePrefix}.column-unique.{$tableName}";
    }

    /**
     * Returns cache-key for caching data related to show create statement of $tableName table.
     *
     * @param string $tableName Name of table to cache,
     * @return string
     */
    private function getShowCreateCacheKey(string $tableName): string {
        return "{$this->cachePrefix}.show-create.{$tableName}";
    }

}
