<?php

namespace InSegment\ApiCore\Services;

use InSegment\ApiCore\Models\SliceRemap;
use InSegment\ApiCore\Models\SliceOperation;

use InSegment\ApiCore\Services\SliceMerger\MergeManager;
use InSegment\ApiCore\Middleware\ChooseVersion;

use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Builder;

class SliceCopy extends SliceOperationBase
{
    /**
     * Initial query
     * 
     * @var \Illuminate\Database\Eloquent\Builder
     */
    private $rootQuery;
    
    /**
     * Array of generated Closures
     * 
     * @var \Closure[]
     */
    private $operations = [];
    
    /**
     * Remaps schema.table of slice operation
     * 
     * @var string 
     */
    private $remapsTable;
    
    /**
     * Remaps table schema of slice operation
     * 
     * @var string 
     */
    private $remapsSchema;
    
    /**
     * Remaps table name without schema of slice operation
     * 
     * @var string 
     */
    private $remapsTblName;
    
    /**
     * Name of query temp table
     *
     * @var string
     */
    private $queryTempTbl;
    
    /**
     * ID of query for consistency checks
     *
     * @var int
     */
    private $queryId = 0;
    
    /**
     * Constructor
     * Initialises operation of copying a slice
     * 
     * @param array $relationsToCopy [
     *     string $relationPath => [
     *         'class' => string $relationOwnerClass,
     *         'remap' => [
     *             string $remappedColumn => string $columnOwnerClass
     *         ]
     *     ],
     * ]
     * @param \Illuminate\Database\Eloquent\Builder $rootQuery
     * @param callable $relationJoinProvider function (\Illuminate\Database\Eloquent\Builder $query, array $relationJoinDefinition, string $joinType)
     */
    public function __construct(array $relationsToCopy, Builder $rootQuery, callable $relationJoinProvider) {
        ChooseVersion::chooseVersion();
        $this->rootQuery = $rootQuery;
        $this->slice = SliceOperation::newSlice();
        $this->mergeManager = $this->slice->getMergeManager();
        $this->remapsTable = $this->slice->getRemapsTable();
        list($this->remapsSchema, $this->remapsTblName) = explode('.', $this->remapsTable);
        $this->queryTempTbl = "temp_{$this->remapsTblName}";

        // Phase 1. Copy all listed tables, no descending remaps yet
        foreach ($relationsToCopy as $pathToRelation => $options) {
            $copyExample = new $options['class'];
            $copyTable = $copyExample->getTable();
            $copyKey = $copyExample->getKeyName();
            
            $this->copyTables[$copyTable] = $copyTable;
            $this->targetTables[$copyTable] = $copyTable;
            $this->uuidableMentions[$copyTable][$copyKey] = $copyKey;
            
            $this->operations[] = function () use ($relationJoinProvider, $pathToRelation, $copyKey, $copyTable) {
                // use query to select replacements for keys found by relation into replacements table
                $this->insertRemaps($relationJoinProvider, $pathToRelation, $copyTable, $copyTable, $copyKey, $copyKey);
                // select entire records by replacement's original identifiers into temp table
                $inserted = $this->selfCopyOperation($copyKey, $copyTable);
                !isset($this->estimateCounters['new'][$copyTable]) && $this->estimateCounters['new'][$copyTable] = 0;
                $this->estimateCounters['written'][$copyTable] = ($this->estimateCounters['new'][$copyTable] += $inserted);
                $this->queryId++;
            };
        }
        
        // Phase 2. Go over all relations again now look at ancestors which have replacements for identifiers
        $uuidablesSort = [];
        $uuidableMergeMentions = [];
        foreach ($relationsToCopy as $pathToRelation => $options) {
            $copyExample = new $options['class'];
            $copyTable = $copyExample->getTable();
            $copyKey = $copyExample->getKeyName();
            $uuidablesSort[$copyTable] = true;
            
            foreach ($options['remap'] as $remapColumn => $targetClass) {
                $targetExample = new $targetClass;
                $targetTable = $targetExample->getTable();
                if (!isset($this->copyTables[$targetTable])) {
                    continue; // we haven't copied anything this column points to
                }
                
                $this->targetTables[$targetTable] = $targetTable;
                $this->uuidableMentions[$copyTable][$remapColumn] = $remapColumn;
                $uuidableMergeMentions[$targetTable][$copyTable][] = $remapColumn;

                $this->operations[] = function () use ($relationJoinProvider, $pathToRelation, $copyKey, $copyTable, $remapColumn, $targetTable) {
                    // use query to select replacements for others tables keys found by relation into replacements table
                    $this->insertRemaps($relationJoinProvider, $pathToRelation, $copyTable, $targetTable, $copyKey, $remapColumn);
                    // update matches in target tables by secondary columns in their respective temporary tables
                    $this->targetUpdateOperation($copyKey, $copyTable, $remapColumn, $targetTable);
                    $this->queryId++;
                };
            }
        }
        
        // do key sorting of merge mentions
        $this->uuidableMergeMentions = array_intersect_key(array_merge($uuidablesSort, $uuidableMergeMentions), $uuidableMergeMentions);
    }
    
    /**
     * Get query selecting data for insert into remaps
     * Then insert remaps from query into remaps table
     * 
     * @param callable $relationJoinProvider
     * @param string $pathToRelation
     * @param string $copyTable
     * @param string $remapTable
     * @param string $queryKey
     * @param string $originalIdColumn
     * @return \Illuminate\Database\Eloquent\Builder
     */
    private function insertRemaps($relationJoinProvider, $pathToRelation, $copyTable, $remapTable, $queryKey, $originalIdColumn)
    {
        $remapsQuery = (clone $this->rootQuery)->select([])->selectRaw("? as `query_id`", [$this->queryId]);

        // use '' for the root definition
        if ($pathToRelation) {
            $relationJoinDefinition = [$pathToRelation => [
                "{$queryKey} as query_key",
                "{$originalIdColumn} as original_id"
            ]];
            $remapsQuery = $relationJoinProvider($remapsQuery, $relationJoinDefinition, 'inner') ?: $remapsQuery;
        } else {
            $remapsQuery->addSelect([
                "{$copyTable}.{$queryKey} as query_key",
                "{$copyTable}.{$originalIdColumn} as original_id"
            ]);
        }
        
        // first, we store all records we've got by relation with query id and their key
        DB::statement("INSERT INTO `{$this->queryTempTbl}` (`query_id`, `query_key`, `original_id`) {$remapsQuery->toSql()}", $remapsQuery->getBindings());
        
        // then insert them into remaps, but they will collapse into less records by original_id which will have only one UUID
        return DB::affectingStatement(
<<<SQL
INSERT INTO `{$this->remapsSchema}`.`{$this->remapsTblName}`
 (`substitution_uuid`, `table`, `query_id`, `original_id`, `destination_id`)
SELECT
 UUID_SHORT() as `substitution_uuid`,
 ? as `table`,
 `{$this->queryTempTbl}`.`query_id`,
 `{$this->queryTempTbl}`.`original_id`,
 `{$this->queryTempTbl}`.`original_id` as `destination_id`
FROM `{$this->queryTempTbl}`
WHERE `{$this->queryTempTbl}`.`query_id` = ?
ON DUPLICATE KEY UPDATE `query_id` = VALUES(`query_id`)
SQL
        , [$remapTable, $this->queryId]);
    }
    
    /**
     * Run operations planned to copy from real tables to temporary ones
     * 
     * @return $this
     */
    public function runOperations()
    {
        $this->prepareOperations();
        foreach ($this->operations as $operation) {
            $operation();
        }

        return $this;
    }
    
    /**
     * Get name of temporary table for model class
     * 
     * @param string $modelClass
     * @return string|null
     */
    public function getTempInsertTable(string $modelClass)
    {
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        return $this->tempTables[$modelTable][MergeManager::TEMP_TABLE_TYPE_INSERT] ?? false;
    }
    
    /**
     * Prepare temporary tables
     * 
     * @return null
     */
    private function prepareOperations()
    {
        SliceRemap::establish([
            'tables' => array_values($this->targetTables),
            'inMemory' => false,
            'uid' => $this->slice->getUID()
        ]);
        
        foreach ($this->copyTables as $copyTable) {
            $this->tempTables[$copyTable] = $this->mergeManager->createTempTable($copyTable, $this->uuidableMentions[$copyTable] ?? [], true);
        }
        
        $this->createQueryTemp();
    }
    
    /**
     * Copy records from real table into temporary table
     * 
     * @param string $copyKey
     * @param string $copyTable
     * @return int
     */
    private function selfCopyOperation($copyKey, $copyTable)
    {
        $columns = array_diff($this->mergeManager->getTableColumns($copyTable), [$copyKey]);
        $selectColumns = array_map(function ($column) use ($copyTable) {
            return "{$copyTable}.{$column}";
        }, $columns);

        $selectColumns[] = "key_remap.substitution_uuid as {$copyKey}";
        $columns[] = $copyKey;

        $copyQuery = DB::table("{$this->remapsTable} as key_remap")
            ->select($selectColumns)
            ->join($copyTable, function ($join) use ($copyKey, $copyTable) {
                return $join
                    ->where('key_remap.table', '=', $copyTable)
                    ->on('key_remap.original_id', '=', "{$copyTable}.{$copyKey}")
                    ->where('key_remap.query_id', '=', $this->queryId);
            });
        
        $insertColumnsImplode = implode(',', array_map(function ($column) {
            return "`{$column}`";
        }, $columns));

        $tempTable = $this->tempTables[$copyTable][MergeManager::TEMP_TABLE_TYPE_INSERT];
        list($tempTableSchema, $tempTblName) = explode('.', $tempTable);

        return DB::affectingStatement(
            "INSERT IGNORE INTO `{$tempTableSchema}`.`{$tempTblName}` ({$insertColumnsImplode}) {$copyQuery->toSql()}",
            $copyQuery->getBindings()
        );
    }
    
    /**
     * Update columns of copied records of one table leading by relation to the copied records of another table
     * 
     * @param string $copyKey
     * @param string $copyTable
     * @param string $remapColumn
     * @param string $targetTable
     * @return int
     */
    private function targetUpdateOperation($copyKey, $copyTable, $remapColumn, $targetTable)
    {
        $tempTable = $this->tempTables[$copyTable][MergeManager::TEMP_TABLE_TYPE_INSERT];
        return DB::table("{$this->queryTempTbl} as query_temp")
            ->where('query_temp.query_id', '=', $this->queryId)
            ->join("{$this->remapsTable} as key_remap", function ($join) use ($targetTable) {
                return $join
                    ->where('key_remap.table', '=', $targetTable)
                    ->on('key_remap.query_id', '=', 'query_temp.query_id')
                    ->on('key_remap.original_id', '=', 'query_temp.original_id');
            })
            ->join("{$this->remapsTable} as copy_remap", function ($join) use ($copyTable) {
                return $join
                    ->where('copy_remap.table', '=', $copyTable)
                    ->on('copy_remap.original_id', '=', 'query_temp.query_key');
            })
            ->join("{$tempTable} as copy_temp", function ($join) use ($copyKey, $remapColumn) {
                return $join
                    ->on("copy_temp.{$copyKey}", '=', 'copy_remap.substitution_uuid')
                    ->on("copy_temp.{$remapColumn}", '=', 'key_remap.original_id');
            })
            ->update(["copy_temp.{$remapColumn}" => DB::raw("`key_remap`.`substitution_uuid`")]);
    }
    
    /**
     * Create query temp-table to hold identifiers of queries
     * 
     * @return null
     */
    private function createQueryTemp()
    {
        DB::statement(
<<<SQL
CREATE TEMPORARY TABLE `{$this->queryTempTbl}` (
  `original_id` int(10) unsigned DEFAULT NULL,
  `query_id` int(10) unsigned NOT NULL,
  `query_key` varchar(255) NOT NULL,
  PRIMARY KEY (`query_id`, `query_key`),
  KEY (`original_id`),
  KEY (`query_key`),
  KEY (`query_id`, `original_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
SQL
        );
    }
    
}
