<?php

namespace InSegment\ApiCore\Services;

use InSegment\ApiCore\Models\UUIDGeneration;
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Models\SliceStructure;
use InSegment\ApiCore\Services\SliceMerger\MergeManager;

class SliceExchange extends SliceOperationBase
{
    /**
     * Class name of auto key remapper
     *
     * @var string 
     */
    protected $autoKeyRemapClass = UUIDGeneration::class;
    
    /**
     * Constructor
     * Initialises operation of exchanging a slice
     * 
     * @param \InSegment\ApiCore\Models\SliceOperation $slice
     * @param array $sliceExchangeStructure [
     *     [
     *         'table' => string $tableName,
     *         'keyName' => string $keyName,
     *         'remap' => [
     *             string $remappedColumn => string $columnOwnerTable
     *         ]
     *     ],
     *     ...
     * ]
     */
    public function __construct(SliceOperation $slice, array $sliceExchangeStructure) {
        $this->slice = $slice;
        $this->mergeManager = $this->slice->getMergeManager([
            MergeManager::TEMP_TABLE_TYPE_INSERT,
            MergeManager::TEMP_TABLE_TYPE_UPDATE,
            MergeManager::TEMP_TABLE_TYPE_CHANGES,
        ]);

        // Phase 1. Copy all listed tables, no descending remaps yet
        foreach ($sliceExchangeStructure as $options) {
            $copyTable = $options['table'];
            $copyKey = $options['keyName'];
            
            $this->copyTables[$copyTable] = $copyTable;
            $this->targetTables[$copyTable] = $copyTable;
            $this->uuidableMentions[$copyTable][$copyKey] = $copyKey;
        }
        
        // Phase 2. Go over all relations again now look at ancestors which have replacements for identifiers
        $uuidablesSort = [];
        $uuidableMergeMentions = [];
        foreach ($sliceExchangeStructure as $options) {
            $copyTable = $options['table'];
            $copyKey = $options['keyName'];
            $uuidablesSort[$copyTable] = true;
            
            foreach ($options['remap'] as $remapColumn => $targetTable) {
                if (!isset($this->copyTables[$targetTable])) {
                    continue; // we haven't copied anything this column points to
                }
                
                $this->targetTables[$targetTable] = $targetTable;
                $this->uuidableMentions[$copyTable][$remapColumn] = $remapColumn;
                $uuidableMergeMentions[$targetTable][$copyTable][] = $remapColumn;
            }
        }
        
        // do key sorting of merge mentions
        $this->uuidableMergeMentions = array_intersect_key(array_merge($uuidablesSort, $uuidableMergeMentions), $uuidableMergeMentions);
    }
    
    /**
     * Run operations planned to copy from real tables to temporary ones
     * CAUTION: Never use this method with user input!
     * 
     * @paray array $counts
     * @param resource|string|\Symfony\Component\Process\InputStream $input
     * @param callable|null $callback
     * @return $this
     */
    public function import(array $counts, $input, $callback = null)
    {
        $uid = $this->slice->getUID();
        $this->estimateCounters = $counts;
        $this->tempTables = $this->mergeManager->listTemporaryTableNames($this->copyTables);
        $structureTemps = [];
        
        foreach ($this->tempTables as $copyTable => $tempsForCopyTable) {
            foreach ($tempsForCopyTable as $type => $tempFullName) {
                // leave only the name of the table but schema - it will be placed upon another server, where schema might differ
                $tableName = explode('.', $tempFullName)[1];
                $structureTemps[$copyTable][$type] = $tableName;
            }
        }
        
        $sliceStructure = new SliceStructure([
            'uid' => $uid,
            'tables' => $structureTemps,
            'sql' => null,
        ]);
        
        $commands = MysqlCommands::fromConnectionConfig();
        $sliceStructure->importTables($commands, $input, $callback);
    }
    
    /**
     * Pack UID, tables and SQL code with tables creation for exchange
     * 
     * @return array [
     *     'uid' => string $uid,
     *     'tables' => array [
     *          string $tableName => [
     *              string $type => string $tempTableOfType
     *          ],
     *          ...
     *     ],
     *     'sql' => string[]
     * ]
     */
    public function packForExchange()
    {
        $uid = $this->slice->getUID();
        $resultSql = [];
        $tempTables = [];
        $putSQL = function ($sql, $type, $table) use (&$resultSql) {
            $resultSql[] = $sql;
        };
        
        foreach ($this->copyTables as $copyTable) {
            $tempsForCopyTable = $this->mergeManager->generateTempStatementSql($copyTable, $this->uuidableMentions[$copyTable] ?? [], true, false, $putSQL);
            
            foreach ($tempsForCopyTable as $type => $tempFullName) {
                // leave only the name of the table but schema - it will be placed upon another server, where schema might differ
                $tableName = explode('.', $tempFullName)[1];
                $tempTables[$copyTable][$type] = $tableName;
            }
        }
        
        return [
            'uid' => $uid,
            'tables' => $tempTables,
            'sql' => $resultSql,
        ];
    }
    
}
