<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Support\Arr;
use InSegment\ApiCore\Exceptions\ApiTransactorException;
use InSegment\ApiCore\Services\ParseCreateTable;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

use InSegment\ApiCore\Interfaces\SliceManagementInterface as SliceManager;

class SliceMerger
{
    // TODO: refactor Transactor to use these constants
    const TEMP_TABLE_TYPE_INSERT = 'insert';
    const TEMP_TABLE_TYPE_INSERT_IGNORE = 'place';
    const TEMP_TABLE_TYPE_UPDATE = 'update';
    const TEMP_TABLE_TYPE_REPLACE = 'replace';
    const TEMP_TABLE_TYPE_CHANGES = 'changes';
    
    const DEFAULT_TEMP_TABLE_TYPES = [
        self::TEMP_TABLE_TYPE_INSERT,
        self::TEMP_TABLE_TYPE_UPDATE,
    ];
    
    const CLEARABLE_TEMP_TABLE_TYPES = [
        self::TEMP_TABLE_TYPE_INSERT,
        self::TEMP_TABLE_TYPE_INSERT_IGNORE,
        self::TEMP_TABLE_TYPE_UPDATE,
        self::TEMP_TABLE_TYPE_REPLACE,
        self::TEMP_TABLE_TYPE_CHANGES,
    ];
    
    /**
     * Show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable 
     */
    private $showCreateParser;
    
    /**
     * Slice manager
     *
     * @var \InSegment\ApiCore\Interfaces\SliceManagementInterface
     */
    private $sliceManager;
    
    /**
     * Temporary table types
     * 
     * @var array
     */
    private $tempTableTypes;
    
    /**
     * Whether this merger uses changes tables
     *
     * @var bool
     */
    private $usesChanges;
    
    /**
     * Store for columns listings for tables
     * 
     * @var array [
     *     $table => string[] $columns,
     *     ...
     * ]
     */
    private $columnStore = [];
    
    /**
     * Store for ODKU statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $ODKUStore = [];
    
    /**
     * Store for ODKU with changes statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $changesODKUStore = [];
    
    /**
     * Store for INSERT statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $insertSelectionStore = [];
    
    /**
     * Cache prefix for slice merger
     *
     * @var string
     */
    private $cachePrefix;
    
    /**
     * Auto-key remap model
     *
     * @var \InSegment\ApiCore\Interfaces\AutoKeyRemapInterface
     */
    private $autoKeyRemapModel;
    
    /**
     * Generations match model
     *
     * @var \InSegment\ApiCore\Models\GenerationsMatch
     */
    private $generationsMatchModel;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Interfaces\SliceManagementInterface $sliceManager
     * @param \InSegment\ApiCore\Services\ParseCreateTable $showCreateParser
     * @param array $tempTableTypes
     */
    public function __construct(SliceManager $sliceManager, ParseCreateTable $showCreateParser, array $tempTableTypes = self::DEFAULT_TEMP_TABLE_TYPES)
    {
        $this->sliceManager = $sliceManager;
        $this->showCreateParser = $showCreateParser;
        $this->tempTableTypes = $tempTableTypes;
        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'slice-merger';
        $this->usesChanges = in_array(self::TEMP_TABLE_TYPE_CHANGES, $tempTableTypes);
    }
    
    /**
     * Create temporary table
     * 
     * @param string $table
     * @param string[] $uuidableTableMentions
     * @param bool $isUuidable
     * @param bool $allowInMemory
     * @return [
     *     string $type => string $tempTableOfType
     * ],
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function createTempTable($table, $uuidableTableMentions, $isUuidable, $allowInMemory = false)
    {
        $useSQL = function ($sql, $type, $table) {
            if (!DB::statement($sql)) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_CREATE_TABLE_FAIL))
                    ->compile($type, $table);
            }
        };
        
        return $this->generateTempStatementSql($table, $uuidableTableMentions, $isUuidable, $allowInMemory, $useSQL);
    }
    
    /**
     * Create temporary table
     * 
     * @param string $table
     * @param string[] $uuidableTableMentions
     * @param bool $isUuidable
     * @param bool $allowInMemory
     * @param callable $useSQL
     * @return [
     *     string $type => string $tempTableOfType
     * ],
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function generateTempStatementSql($table, $uuidableTableMentions, $isUuidable, $allowInMemory, callable $useSQL)
    {
        $createStatementSplit = $this->showCreateParser->getCreateForTemp($table, $uuidableTableMentions, $isUuidable, $allowInMemory, false);
        if ($this->usesChanges) {
            $primaryKey = $this->showCreateParser->getPrimaryKeyData($table);
            if ($primaryKey === null) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_CHANGES_NEED_PRIMARY_KEY))->compile($table);
            }
            
            $changesStatementSplit = $this->showCreateParser->getCreateForTemp($table, $uuidableTableMentions, $isUuidable, $allowInMemory, true);
        }
        
        $tempTables = [];
        
        // first, create temporary tables
        foreach ($this->tempTableTypes as $type) {
            list($tempTableDb, $tempTable) = explode('.', $tempTables[$type] = $this->prefixTableForTransaction($table, $type));
            
            if ($type === self::TEMP_TABLE_TYPE_CHANGES) {
                $changesStatementSplit[0] = "CREATE TABLE `{$tempTableDb}`.`{$tempTable}` (";
                $createStatementTemp = implode("\n", $changesStatementSplit);
            } else {
                $createStatementSplit[0] = "CREATE TABLE `{$tempTableDb}`.`{$tempTable}` (";
                $createStatementTemp = implode("\n", $createStatementSplit);
            }
            
            $useSQL($createStatementTemp, $type, $table);
        }
        
        return $tempTables;
    }
    
    /**
     * Drop temporary table
     * 
     * @param array $tempTableDef [
     *     string $type => string $tempTableOfType
     * ]
     * @return null
     */
    public function dropTempTable(array $tempTableDef)
    {
        foreach ($this->tempTableTypes as $type) {
            list ($tempTableDb, $tempTableName) = explode('.', $tempTableDef[$type]);
            DB::statement("DROP TABLE IF EXISTS `{$tempTableDb}`.`{$tempTableName}`");
        }
    }
    
    /**
     * Merge temporary tables into real ones
     * 
     * @param array $options [
     *     'tempTables' => &[ // enumeration of temporary tables
     *         string $tableName => [
     *             string $type => string $tempTableOfType,
     *             ...
     *         ],
     *         ...
     *     ],
     *     'uuidableMentions' => [ // tables to the keys which are mentioned in some uuidable tables and thus may have updated key
     *         string $tableName => string|string[] $keys,
     *         ...
     *     ],
     *     'deletionEnabledTables' => array|null,
     *     'postProcessing' => callable function (\InSegment\ApiCore\Interfaces\SliceManagementInterface $sliceManager),
     *     'supplementaries' => [
     *         'autoKeyRemap' => string $autoKeyRemapClass, // the Model for table which contains map of UUID to the new identifiers
     *         'generationsMatch' => string $generationsMatchClass // the Model for matching the new identifiers in order
     *         'deletes' => string|null $deletesModelClass, // this Model's table lists records identifiers to be deleted from real tables
     *         'logs' => string|null $logsModelClass, // this Model's table with transacted records identifiers logged for use in post-processing
     *     ]
     * ]
     * @return null
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function mergeTempTables(array $options)
    {
        $currentTable = null;
        $currentOp = null;
        
        try {
            DB::transaction(function () use (&$options, &$currentTable, &$currentOp) {
                $uid = $this->sliceManager->getUID();
                $autoKeyRemapClass = $options['supplementaries']['autoKeyRemap'];
                $generationsMatchClass = $options['supplementaries']['generationsMatch'];
                $deletesModelClass = Arr::get($options['supplementaries'], 'deletes');
                $logsModelClass = Arr::get($options['supplementaries'], 'logs');
                
                $this->autoKeyRemapModel = (new $autoKeyRemapClass([], $uid));
                $this->autoKeyRemapModel->disposeUnused();
                $this->generationsMatchModel = new $generationsMatchClass();
                $deletesTable = $deletesModelClass ? (new $deletesModelClass([], $uid))->getTable() : null;
                $logsTable = $logsModelClass ? (new $logsModelClass([], $uid))->getTable() : null;
                
                if (isset($deletesTable) && count(Arr::get($options, 'deletionEnabledTables', []))) {
                    $currentOp = 'groupsForDelete';
                    $currentTable = $deletesTable;
                    $groupsForDelete = $deletesModelClass::groupBy('table')->pluck('key_name', 'table')->toArray();
                } else {
                    $groupsForDelete = [];
                }
                
                // The generations table contains map of UUID to some table, and real id in the real table,
                // which is, initially, equals to NULL, until the table is merged.
                // 
                // As we cannot allocate auto-increment primary keys in advance before we insert the rows,
                // we must call LAST_INSERT_ID() to retrieve the first inserted id. All ids of mass INSERT
                // are consecutive when innodb_autoinc_lock_mode = 1 (this should be configured on the db server)
                // and so we can determine auto-increment key of N'th inserted as LAST_INSERT_ID() + (N - 1)
                // 
                // As we cannot properly relate the rows in advance not knowing their ids, the order of insertion
                // must be precomputed in uuidables.json, and we must insert first the rows, which do not need any
                // not inserted relations, and which are needed by relations to be inserted later
                
                $depends = [];
                foreach ($options['uuidableMentions'] as $table => $dependents) {
                    if (isset($options['tempTables'][$table])) {
                        foreach ($dependents as $dependentTable => $dependentKeys) {
                            $dependentKeysArray = Arr::wrap($dependentKeys);
                            
                            if (isset($depends[$dependentTable][$table])) {
                                $depends[$dependentTable][$table] = array_unique(array_merge($depends[$dependentTable][$table], $dependentKeysArray));
                            } else {
                                $depends[$dependentTable][$table] = $dependentKeysArray;
                            }
                        }
                    }
                }
                
                $movableTableTypes = array_diff($this->tempTableTypes, [self::TEMP_TABLE_TYPE_CHANGES]);
                $mergedTables = [];
                $affectedByInsert = [];
                foreach ($options['tempTables'] as $table => $tempTable) {
                    $currentTable = $table;
                    $currentOp = 'dependencies';
                    
                    // Tables which are necessary for that table to be merged
                    $mapTableDependsOn = $depends[$table] ?? [];
                    $needToUpdateIdentifiers = [];
                    
                    // Determine, whether there was any insert into any table this table is dependent on
                    foreach ($mapTableDependsOn as $dependencyTable => $dependentKeys) {
                        if (!isset($mergedTables[$dependencyTable])) {
                            throw (new ApiTransactorException(ApiTransactorException::CODE_UUIDABLES_WRONG_ORDER))
                                ->compile('`' . collect($dependentKeys)->implode('`, `') . '`', $table, $dependencyTable);
                        }
                        
                        if ($affectedByInsert[$dependencyTable] > 0) {
                            foreach ($dependentKeys as $key) {
                                $needToUpdateIdentifiers[$key] = true;
                            }
                        }
                    }
                    
                    // If there was a result of determination, update exactly one time each needed key
                    if (count($needToUpdateIdentifiers)) {
                        $currentOp = 'relations';
                        foreach (array_keys($needToUpdateIdentifiers) as $dependentKey) {
                            foreach ($movableTableTypes as $type) {
                                $this->updateRelationsInTemp($tempTable[$type], $dependentKey);
                            }
                        }
                    }
                    
                    // If there were deletions from the table, perform then now
                    if (isset($deletesTable, $groupsForDelete[$table])) {
                        $this->performTableDeletions($deletesTable, $table, $groupsForDelete[$table]);
                    }
                    
                    // Merge temporary tables into real ones
                    $affectedByInsert[$table] = 0;
                    foreach ($movableTableTypes as $type) {
                        $currentOp = $type;
                        $affected = $this->doMoveTable($table, $tempTable, $type);
                        
                        if (in_array($type, [self::TEMP_TABLE_TYPE_INSERT, self::TEMP_TABLE_TYPE_INSERT_IGNORE])) {
                            $affectedByInsert[$table] += $affected;
                        }
                    }
                    
                    // If some of the inserted records were deleted later in transaction
                    if (isset($deletesTable, $groupsForDelete[$table]) && $affectedByInsert[$table]) {
                        $currentOp = 'delete';
                        
                        // First update deletions table and have count of changes
                        if (($updatedDeletions = $this->updateDeletionsInTemp($deletesTable, $table)) > 0) {
                            // If there was any changed deletions, repeat the query to delete from real table
                            $this->performTableDeletions($deletesTable, $table, $groupsForDelete[$table]);
                        }
                    }
                    
                    $mergedTables[$table] = true;
                }
                
                // Update logs table so it can be used in post-processing to determine rows affected by transaction
                if (isset($logsTable)) {
                    $currentOp = 'logsRelations';
                    $currentTable = $logsTable;
                    $this->updateRelationsInTemp($logsTable, 'key', true);
                }
                
                $currentOp = 'postProcess';
                DB::statement("SET FOREIGN_KEY_CHECKS=1");
                $options['postProcessing']($this->sliceManager);
                DB::statement("SET FOREIGN_KEY_CHECKS=0");
            });
        } catch (\Throwable $exception) {
            if ($currentOp === 'postProcess') {
                throw (new ApiTransactorException(ApiTransactorException::CODE_POST_PROCESS_FAILED, $exception))->compile();
            } else {
                throw (new ApiTransactorException(ApiTransactorException::CODE_FAIL_UNDONE, $exception))
                    ->compile($currentOp, $currentTable);
            }
        }
    }
    
    /**
     * Get a list of table names (not necessarily created ones!) for temporary tables
     * 
     * @param string[] $tableNames
     * @return [
     *     string $tableName => [
     *         string $type => string $tempTableOfType,
     *         ...
     *     ],
     *     ...
     * ]
     */
    public function listTemporaryTableNames($tableNames)
    {
        $result = [];
        
        foreach ($tableNames as $tableName) {
            foreach ($this->tempTableTypes as $type) {
                $tempTableOfType = $this->prefixTableForTransaction($tableName, $type);
                $result[$tableName][$type] = $tempTableOfType;
            }
        }
        
        return $result;
    }
    
    /**
     * REPLACE statement from transaction table
     * 
     * @param string $realTable
     * @param array $tempTable [string $type => string $tempTableName]
     * @param string $type
     * @param int $shouldBeReplaced
     * @return int
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    private function doMoveTable($realTable, $tempTable, $type)
    {
        $moveTable = $tempTable[$type];
        list($database, $moveTableName) = explode('.', $moveTable);
        $quotedTempTable = "`{$database}`.`{$moveTableName}`";
        
        switch ($type) {
            case self::TEMP_TABLE_TYPE_INSERT:
                $insertSelection = $this->getInsertSelection($realTable, $quotedTempTable);
                $estimate = $this->sliceManager->getNewCount($realTable);
                
                // do not risk and waste time on nothing
                if ($estimate === 0) {
                    return;
                }
                
                $affected = DB::affectingStatement($op = "INSERT INTO `{$realTable}` {$this->getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)}");
                $this->updateGenerations($realTable, $moveTable, $insertSelection, $affected);
            break;
            case self::TEMP_TABLE_TYPE_INSERT_IGNORE:
                $insertSelection = $this->getInsertSelection($realTable, $quotedTempTable);
                $estimate = null;
                $affected = DB::affectingStatement($op = "INSERT IGNORE INTO `{$realTable}` {$this->getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)}");
                $this->updateGenerations($realTable, $moveTable, $insertSelection, $affected);
            break;
            case self::TEMP_TABLE_TYPE_UPDATE:
                $estimate = null;
                if ($this->usesChanges) {
                    $changesTable = $tempTable[self::TEMP_TABLE_TYPE_CHANGES];
                    list($changesDatabase, $changesTableName) = explode('.', $changesTable);
                    $quotedChangesTable = "`{$changesDatabase}`.`{$changesTableName}` as `current_update_table_changes`";
                    $primaryKey = $this->showCreateParser->getPrimaryKeyData($realTable);
                    $changesJoin = ["INNER JOIN {$quotedChangesTable} ON "];
                    foreach ($primaryKey['columns'] as $changesJoinColumn) {
                        $changesJoin[] = "`current_update_table_changes`.`{$changesJoinColumn}` = {$quotedTempTable}.`{$changesJoinColumn}`";
                    }
                    
                    $implodeChangesJoin = implode("\n  ", $changesJoin);
                    $odku = $this->getChangesODKUForTable($realTable);
                    $op = "INSERT INTO `{$realTable}` SELECT {$quotedTempTable}.* FROM {$quotedTempTable}\n{$implodeChangesJoin}\nON DUPLICATE KEY UPDATE {$odku}";
                } else {
                    $odku = $this->getODKUForTable($realTable);
                    $op = "INSERT INTO `{$realTable}` SELECT * FROM {$quotedTempTable} ON DUPLICATE KEY UPDATE {$odku}";
                }
                $affected = DB::affectingStatement($op);
            break;
            case self::TEMP_TABLE_TYPE_REPLACE:
                $estimate = 2 * ($this->sliceManager->getWrittenCount($realTable) - $this->sliceManager->getNewCount($realTable));
                
                // do not risk and waste time on nothing
                if ($estimate === 0) {
                    return;
                }
                
                $affected = DB::affectingStatement($op = "REPLACE INTO `{$realTable}` SELECT * FROM {$quotedTempTable}");
            break;
            default: throw (new ApiTransactorException(ApiTransactorException::CODE_UNKNOWN_TYPE_OF_MOVE_OPERATION))->compile($type);
        }
        
        if (isset($estimate) && $affected !== $estimate) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_UNEXPECTED_CASUALTIES))
                ->compile($type, $moveTableName, $affected, $op, $estimate);
        }
        
        return $affected;
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table
     * 
     * @param string $table
     * @return string
     */
    public function getTableColumns($table)
    {
        if (!isset($this->columnStore[$table])) {
            $cacheKey = "{$this->cachePrefix}.columns.{$table}";
            $this->columnStore[$table] = Cache::rememberForever($cacheKey, function () use ($table) {
                return DB::getSchemaBuilder()->getColumnListing($table);
            });
        }
        
        return $this->columnStore[$table];
    }
    
    /**
     * Update UUID generations after insertion is done
     * 
     * @param string $realTable
     * @param string $tempTable
     * @param array [string|null $implodeEnumeration, string|null $autoKey] $insertSelection
     * @param int $affectedByInsert
     * @return int
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    private function updateGenerations($realTable, $tempTable, $insertSelection, $affectedByInsert)
    {
        $autoKey = $insertSelection[1];
        
        if ($autoKey === null || $affectedByInsert === 0) {
            return;
        }
        
        $affectedGenerations = $this->generationsMatchModel->match($this->autoKeyRemapModel, $realTable, $tempTable, $autoKey);
        
        if ($affectedGenerations !== $affectedByInsert) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_AFFECTED_LESS_GENERATIONS_THAN_INSERTED))
                ->compile($realTable, $affectedGenerations, $affectedByInsert);
        }
        
        return $affectedByInsert;
    }
    
    /**
     * Update UUID's in the temporary table relating to auto-keys of some real tables through remapper
     * 
     * @param string $tempTable
     * @param string $key
     * @param bool $isStringKey
     * @return int
     */
    private function updateRelationsInTemp(string $tempTable, string $key, bool $isStringKey = false)
    {
        $uuidField = $this->autoKeyRemapModel->uuidFieldName();
        $autoKeyField = $this->autoKeyRemapModel->autoKeyFieldName();

        if ($isStringKey) {
            $joinKey = $this->castFieldStringAsUnsigned("`temp_table`.`{$key}`");
        } else {
            $joinKey = "temp_table.{$key}";
        }

        return DB
            ::table("{$tempTable} as temp_table")
            ->join("{$this->autoKeyRemapModel->getTable()} as gens_table", "gens_table.{$uuidField}", '=', $joinKey)
            ->update(["temp_table.{$key}" => DB::raw("`gens_table`.`{$autoKeyField}`")]);
    }
    
    /**
     * Update UUID's in the temporary table deleting remapped auto-keys of some real tables
     * 
     * @param string $deletionsTable
     * @param string $table
     * @return int
     */
    private function updateDeletionsInTemp(string $deletionsTable, string $table)
    {
        $uuidField = $this->autoKeyRemapModel->uuidFieldName();
        $autoKeyField = $this->autoKeyRemapModel->autoKeyFieldName();
        $joinKey = $this->castFieldStringAsUnsigned('`dels_table`.`key`');

        return DB
            ::table("{$deletionsTable} as dels_table")
            ->where('dels_table.table', '=', $table)
            ->join("{$this->autoKeyRemapModel->getTable()} as gens_table", "gens_table.{$uuidField}", '=', $joinKey)
            ->update(["dels_table.key" => DB::raw("`gens_table`.`{$autoKeyField}`")]);
    }

    /**
     * Delete rows from $table which keys are present in deletions table
     * 
     * @param string $deletionsTable
     * @param string $table
     * @param string $tableKeyForDeletion
     * @return int
     */
    private function performTableDeletions(string $deletionsTable, string $table, string $tableKeyForDeletion)
    {
        return DB
            ::table($table)
            ->join("{$deletionsTable} as deletes", function ($join) use ($table, $tableKeyForDeletion) {
                return $join->where('deletes.table', '=', $table)->on('deletes.key', '=', "{$table}.{$tableKeyForDeletion}");
            })
            ->delete();
    }
    
    /**
     * Get SELECT part of statement for moving from insert temporary table
     * 
     * @param string $quotedTempTable
     * @param array [string|null $implodeEnumeration, string|null $autoKey] $insertSelection
     * @return string
     */
    private function getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)
    {
        list($implodeEnumeration, $autoKey) = $insertSelection;
        
        if ($autoKey === null) {
            return "SELECT * FROM {$quotedTempTable}";
        } else {
            return "({$implodeEnumeration}) SELECT {$implodeEnumeration} FROM {$quotedTempTable} ORDER BY `{$autoKey}` ASC";
        }
    }
    
    /**
     * Get SELECT listing for INSERT from temporary table to $table
     * 
     * @param string $table
     * @return string
     */
    private function getInsertSelection($table, $quotedTempTable)
    {
        if (!isset($this->insertSelectionStore[$table])) {
            $cacheKey = "{$this->cachePrefix}.updateSet.{$table}";
            $this->insertSelectionStore[$table] = Cache::rememberForever($cacheKey, function () use ($table) {
                $autoKey = $this->showCreateParser->getAutoKey($table);
                
                if ($autoKey === null) {
                    return [null, null];
                }

                $listing = $this->getTableColumns($table);
                $enumeration = [];
                foreach ($listing as $column) {
                    if ($column !== $autoKey) {
                        $enumeration[] = "`{$column}`";
                    }
                }
                
                $implodeEnumeration = implode(', ', $enumeration);
                return [$implodeEnumeration, $autoKey];
            });
        }
        
        return $this->insertSelectionStore[$table];
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table
     * 
     * @param string $table
     * @return string
     */
    private function getODKUForTable($table)
    {
        if (!isset($this->ODKUStore[$table])) {
            $cacheKey = "{$this->cachePrefix}.odku.{$table}";
            $this->ODKUStore[$table] = Cache::rememberForever($cacheKey, function () use ($table) {
                $listing = $this->getTableColumns($table);
                return implode(',', array_map(function ($column) {
                    return "`{$column}` = VALUES(`{$column}`)";
                }, $listing));
            });
        }
        
        return $this->ODKUStore[$table];
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table using a join with changes
     * 
     * @param string $table
     * @return string
     */
    private function getChangesODKUForTable($table)
    {
        if (!isset($this->changesODKUStore[$table])) {
            $cacheKey = "{$this->cachePrefix}.changes-odku.{$table}";
            $this->changesODKUStore[$table] = Cache::rememberForever($cacheKey, function () use ($table) {
                $listing = $this->getTableColumns($table);
                return implode(',', array_map(function ($column) use ($table) {
                    return "`{$column}` = IF(`current_update_table_changes`.`{$column}`, VALUES(`{$column}`), `{$table}`.`{$column}`)";
                }, $listing));
            });
        }
        
        return $this->changesODKUStore[$table];
    }
    
    /**
     * Prefixes table name for transaction table
     * 
     * @param string $tableName
     * @param string $type
     * @return array [string|null $implodeEnumeration, string|null $autoKey]
     */
    private function prefixTableForTransaction(string $tableName, string $type): string
    {
        $database = $this->sliceManager->getSchema();
        $uid = $this->sliceManager->getUID();
        $ret = "{$tableName}_{$type}_{$uid}";
        
        // limit of table name length is 64
        if (strlen($ret) > 64) {
            return substr($ret, 0, 64);
        }
        
        return "{$database}.{$ret}";
    }

    /**
     * Cast string field as unsigned to fix mariaDB issue:
     * "In other cases, arguments are compared as floating point, or real, numbers."
     * @see https://mariadb.com/kb/en/library/type-conversion/#rules-for-conversion-on-comparison
     *
     * @param string $field
     * @return \Illuminate\Database\Query\Expression
     */
    private function castFieldStringAsUnsigned(string $field)
    {
        return DB::raw("CAST({$field} AS UNSIGNED)");
    }
}
