<?php

namespace InSegment\ApiCore\Services\SliceMerger;

use Illuminate\Support\Facades\DB;

use InSegment\ApiCore\Exceptions\ApiTransactorException;

class CopyRecords
{
    /**
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeOperation
     */
    protected $mergeOperation;
    
    /**
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeManager
     */
    protected $mergeManager;
    
    /**
     * Slice manager
     *
     * @var \InSegment\ApiCore\Interfaces\SliceManagementInterface
     */
    protected $sliceManager;

    /**
     * Show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable 
     */
    protected $showCreateParser;
    
    /**
     * @var \Illuminate\Database\Eloquent\Model
     */
    protected $generationsMatchModel;
    
    /**
     * @var \Illuminate\Database\Eloquent\Model
     */
    protected $autoKeyRemapModel;
    
    /**
     * Store for ODKU statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $ODKUStore = [];
    
    /**
     * Store for ODKU with changes statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $changesODKUStore = [];
    
    /**
     * Store for INSERT statement parts
     * 
     * @var array [
     *     $table => string $odku,
     *     ...
     * ]
     */
    private $insertSelectionStore = [];
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Services\SliceMerger\MergeOperation $mergeOperation
     */
    public function __construct(MergeOperation $mergeOperation)
    {
        $this->mergeOperation = $mergeOperation;
        $this->mergeManager = $mergeOperation->getMergeManager();
        $this->sliceManager = $this->mergeManager->getSliceManager();
        $this->showCreateParser = $this->mergeManager->getShowCreateParser();
        $this->generationsMatchModel = $mergeOperation->getGenerationsMatchModel();
        $this->autoKeyRemapModel = $mergeOperation->getAutoKeyRemapModel();
    }
    
    /**
     * Move records from temporary table to real table
     * 
     * @param string $realTable
     * @param array $tempTable [string $type => string $tempTableName]
     * @param string $type
     * @return int
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function moveRecordsFromTempToRealTable($realTable, $tempTable, $type)
    {
        $moveTable = $tempTable[$type];
        list($database, $moveTableName) = explode('.', $moveTable);
        $quotedTempTable = "`{$database}`.`{$moveTableName}`";
        
        switch ($type) {
            case MergeManager::TEMP_TABLE_TYPE_INSERT:
                $insertSelection = $this->getInsertSelection($realTable);
                $estimate = $this->sliceManager->getNewCount($realTable);
                
                // do not risk and waste time on nothing
                if ($estimate === 0) {
                    return;
                }
                
                $affected = DB::affectingStatement($op = "INSERT INTO `{$realTable}` {$this->getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)}");
                $this->updateGenerations($realTable, $moveTable, $insertSelection, $affected);
            break;
            case MergeManager::TEMP_TABLE_TYPE_INSERT_IGNORE:
                $insertSelection = $this->getInsertSelection($realTable);
                $estimate = null;
                $affected = DB::affectingStatement($op = "INSERT IGNORE INTO `{$realTable}` {$this->getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)}");
                $this->updateGenerations($realTable, $moveTable, $insertSelection, $affected);
            break;
            case MergeManager::TEMP_TABLE_TYPE_UPDATE:
                $estimate = null;
                if ($this->mergeManager->getIsUsingChanges()) {
                    $changesTable = $tempTable[MergeManager::TEMP_TABLE_TYPE_CHANGES];
                    list($changesDatabase, $changesTableName) = explode('.', $changesTable);
                    $quotedChangesTable = "`{$changesDatabase}`.`{$changesTableName}` as `current_update_table_changes`";
                    $primaryKey = $this->showCreateParser->getPrimaryKeyData($realTable);
                    $changesJoin = ["INNER JOIN {$quotedChangesTable} ON "];
                    foreach ($primaryKey['columns'] as $changesJoinColumn) {
                        $changesJoin[] = "`current_update_table_changes`.`{$changesJoinColumn}` = {$quotedTempTable}.`{$changesJoinColumn}`";
                    }
                    
                    $implodeChangesJoin = implode("\n  ", $changesJoin);
                    $odku = $this->getChangesODKUForTable($realTable);
                    $op = "INSERT INTO `{$realTable}` SELECT {$quotedTempTable}.* FROM {$quotedTempTable}\n{$implodeChangesJoin}\nON DUPLICATE KEY UPDATE {$odku}";
                } else {
                    $odku = $this->getODKUForTable($realTable);
                    $op = "INSERT INTO `{$realTable}` SELECT * FROM {$quotedTempTable} ON DUPLICATE KEY UPDATE {$odku}";
                }
                $affected = DB::affectingStatement($op);
            break;
            case MergeManager::TEMP_TABLE_TYPE_REPLACE:
                $estimate = 2 * ($this->sliceManager->getWrittenCount($realTable) - $this->sliceManager->getNewCount($realTable));
                
                // do not risk and waste time on nothing
                if ($estimate === 0) {
                    return;
                }
                
                $affected = DB::affectingStatement($op = "REPLACE INTO `{$realTable}` SELECT * FROM {$quotedTempTable}");
            break;
            default: throw (new ApiTransactorException(ApiTransactorException::CODE_UNKNOWN_TYPE_OF_MOVE_OPERATION))->compile($type);
        }
        
        if (isset($estimate) && $affected !== $estimate) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_UNEXPECTED_CASUALTIES))
                ->compile($type, $moveTableName, $affected, $op, $estimate);
        }
        
        return $affected;
    }
    
    /**
     * Get SELECT listing for INSERT from temporary table to $table
     * 
     * @param string $table
     * @return array [string|null $implodeEnumeration, string|null $autoKey]
     */
    private function getInsertSelection($table)
    {
        if (!isset($this->insertSelectionStore[$table])) {
            $this->insertSelectionStore[$table] = $this->mergeManager->prefixedCache("updateSet.{$table}", function () use ($table) {
                $autoKey = $this->showCreateParser->getAutoKey($table);
                
                if ($autoKey === null) {
                    return [null, null];
                }

                $listing = $this->mergeManager->getTableColumns($table);
                $enumeration = [];
                foreach ($listing as $column) {
                    if ($column !== $autoKey) {
                        $enumeration[] = "`{$column}`";
                    }
                }
                
                $implodeEnumeration = implode(', ', $enumeration);
                return [$implodeEnumeration, $autoKey];
            });
        }
        
        return $this->insertSelectionStore[$table];
    }
    
    /**
     * Update UUID generations after insertion is done
     * 
     * @param string $realTable
     * @param string $tempTable
     * @param array [string|null $implodeEnumeration, string|null $autoKey] $insertSelection
     * @param int $affectedByInsert
     * @return int
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    private function updateGenerations($realTable, $tempTable, $insertSelection, $affectedByInsert)
    {
        $autoKey = $insertSelection[1];
        
        if ($autoKey === null || $affectedByInsert === 0) {
            return;
        }
        
        $affectedGenerations = $this->generationsMatchModel->match($this->autoKeyRemapModel, $realTable, $tempTable, $autoKey);
        
        if ($affectedGenerations !== $affectedByInsert) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_AFFECTED_LESS_GENERATIONS_THAN_INSERTED))
                ->compile($realTable, $affectedGenerations, $affectedByInsert);
        }
        
        return $affectedByInsert;
    }
    
    /**
     * Get SELECT part of statement for moving from insert temporary table
     * 
     * @param string $quotedTempTable
     * @param array [string|null $implodeEnumeration, string|null $autoKey] $insertSelection
     * @return string
     */
    private function getSelectFromInsertTempSQL($quotedTempTable, $insertSelection)
    {
        list($implodeEnumeration, $autoKey) = $insertSelection;
        
        if ($autoKey === null) {
            return "SELECT * FROM {$quotedTempTable}";
        } else {
            return "({$implodeEnumeration}) SELECT {$implodeEnumeration} FROM {$quotedTempTable} ORDER BY `{$autoKey}` ASC";
        }
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table
     * 
     * @param string $table
     * @return string
     */
    private function getODKUForTable($table)
    {
        if (!isset($this->ODKUStore[$table])) {
            $this->ODKUStore[$table] = $this->mergeManager->prefixedCache("odku.{$table}", function () use ($table) {
                $listing = $this->mergeManager->getTableColumns($table);
                return implode(',', array_map(function ($column) {
                    return "`{$column}` = VALUES(`{$column}`)";
                }, $listing));
            });
        }
        
        return $this->ODKUStore[$table];
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table using a join with changes
     * 
     * @param string $table
     * @return string
     */
    private function getChangesODKUForTable($table)
    {
        if (!isset($this->changesODKUStore[$table])) {
            $this->changesODKUStore[$table] = $this->mergeManager->prefixedCache("changes-odku.{$table}", function () use ($table) {
                $listing = $this->mergeManager->getTableColumns($table);
                return implode(',', array_map(function ($column) use ($table) {
                    return "`{$column}` = IF(`current_update_table_changes`.`{$column}`, VALUES(`{$column}`), `{$table}`.`{$column}`)";
                }, $listing));
            });
        }
        
        return $this->changesODKUStore[$table];
    }
        
}
