<?php

namespace InSegment\ApiCore\Services\SliceMerger;

use InSegment\ApiCore\Exceptions\ApiTransactorException;
use InSegment\ApiCore\Services\ParseCreateTable;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

use InSegment\ApiCore\Interfaces\SliceManagementInterface;

class MergeManager
{
    /** @var int */
    const INDENTIFIER_MAX_LENGTH = 64;

    // TODO: refactor Transactor to use these constants
    const TEMP_TABLE_TYPE_INSERT = 'insert';
    const TEMP_TABLE_TYPE_INSERT_IGNORE = 'place';
    const TEMP_TABLE_TYPE_UPDATE = 'update';
    const TEMP_TABLE_TYPE_REPLACE = 'replace';
    const TEMP_TABLE_TYPE_CHANGES = 'changes';
    
    const DEFAULT_TEMP_TABLE_TYPES = [
        self::TEMP_TABLE_TYPE_INSERT,
        self::TEMP_TABLE_TYPE_UPDATE,
    ];
    
    const CLEARABLE_TEMP_TABLE_TYPES = [
        self::TEMP_TABLE_TYPE_INSERT,
        self::TEMP_TABLE_TYPE_INSERT_IGNORE,
        self::TEMP_TABLE_TYPE_UPDATE,
        self::TEMP_TABLE_TYPE_REPLACE,
        self::TEMP_TABLE_TYPE_CHANGES,
    ];
    
    /**
     * Show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable 
     */
    private $showCreateParser;
    
    /**
     * Slice manager
     *
     * @var \InSegment\ApiCore\Interfaces\SliceManagementInterface
     */
    private $sliceManager;
    
    /**
     * Temporary table types
     * 
     * @var array
     */
    private $tempTableTypes;
    
    /**
     * Whether this merger uses changes tables
     *
     * @var bool
     */
    private $usesChanges;
    
    /**
     * Store for columns listings for tables
     * 
     * @var array [
     *     $table => string[] $columns,
     *     ...
     * ]
     */
    private $columnStore = [];
    
    /**
     * Cache prefix for slice merger
     *
     * @var string
     */
    private $cachePrefix;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Interfaces\SliceManagementInterface $sliceManager
     * @param \InSegment\ApiCore\Services\ParseCreateTable $showCreateParser
     * @param array $tempTableTypes
     */
    public function __construct(SliceManagementInterface $sliceManager, ParseCreateTable $showCreateParser, array $tempTableTypes = self::DEFAULT_TEMP_TABLE_TYPES)
    {
        $this->sliceManager = $sliceManager;
        $this->showCreateParser = $showCreateParser;
        $this->tempTableTypes = $tempTableTypes;
        $this->cachePrefix = config('api_core.cache_prefix', 'insegment.api-core.') . 'slice-merger';
        $this->usesChanges = in_array(self::TEMP_TABLE_TYPE_CHANGES, $tempTableTypes);
    }
    
    /**
     * @return \InSegment\ApiCore\Services\SliceMerger\SliceManagementInterface
     */
    public function getSliceManager(): SliceManagementInterface
    {
        return $this->sliceManager;
    }
    
    /**
     * @return \InSegment\ApiCore\Services\ParseCreateTable 
     */
    public function getShowCreateParser(): ParseCreateTable
    {
        return $this->showCreateParser;
    }
    
    /**
     * @return array
     */
    public function getMovableTableTypes(): array
    {
        return array_diff($this->tempTableTypes, [self::TEMP_TABLE_TYPE_CHANGES]);
    }
    
    /**
     * @return bool
     */
    public function getIsUsingChanges(): bool
    {
        return $this->usesChanges;
    }
    
    /**
     * @param \InSegment\ApiCore\Services\SliceMerger\MergeOptions $operationOptions
     * @return \InSegment\ApiCore\Services\SliceMerger\MergeOperation
     */
    public function createOperation(MergeOptions $operationOptions): MergeOperation
    {
        return new MergeOperation($this, $operationOptions);
    }
    
    /**
     * Create temporary table
     * 
     * @param string $table
     * @param string[] $uuidableTableMentions
     * @param bool $isUuidable
     * @param bool $allowInMemory
     * @return [
     *     string $type => string $tempTableOfType
     * ],
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function createTempTable($table, $uuidableTableMentions, $isUuidable, $allowInMemory = false)
    {
        $useSQL = function ($sql, $type, $table) {
            if (!DB::statement($sql)) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_CREATE_TABLE_FAIL))
                    ->compile($type, $table);
            }
        };
        
        return $this->generateTempStatementSql($table, $uuidableTableMentions, $isUuidable, $allowInMemory, $useSQL);
    }
    
    /**
     * Create temporary table
     * 
     * @param string $table
     * @param string[] $uuidableTableMentions
     * @param bool $isUuidable
     * @param bool $allowInMemory
     * @param callable $useSQL
     * @return [
     *     string $type => string $tempTableOfType
     * ],
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function generateTempStatementSql($table, $uuidableTableMentions, $isUuidable, $allowInMemory, callable $useSQL)
    {
        $createStatementSplit = $this->showCreateParser->getCreateForTemp($table, $uuidableTableMentions, $isUuidable, $allowInMemory, false);
        if ($this->usesChanges) {
            $primaryKey = $this->showCreateParser->getPrimaryKeyData($table);
            if ($primaryKey === null) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_CHANGES_NEED_PRIMARY_KEY))->compile($table);
            }
            
            $changesStatementSplit = $this->showCreateParser->getCreateForTemp($table, $uuidableTableMentions, $isUuidable, $allowInMemory, true);
        }
        
        $tempTables = [];
        
        // first, create temporary tables
        foreach ($this->tempTableTypes as $type) {
            list($tempTableDb, $tempTable) = explode('.', $tempTables[$type] = $this->prefixTableForTransaction($table, $type));
            
            if ($type === self::TEMP_TABLE_TYPE_CHANGES) {
                $changesStatementSplit[0] = "CREATE TABLE `{$tempTableDb}`.`{$tempTable}` (";
                $createStatementTemp = implode("\n", $changesStatementSplit);
            } else {
                $createStatementSplit[0] = "CREATE TABLE `{$tempTableDb}`.`{$tempTable}` (";
                $createStatementTemp = implode("\n", $createStatementSplit);
            }
            
            $useSQL($createStatementTemp, $type, $table);
        }
        
        return $tempTables;
    }
    
    /**
     * Drop temporary table
     * 
     * @param array $tempTableDef [
     *     string $type => string $tempTableOfType
     * ]
     * @return null
     */
    public function dropTempTable(array $tempTableDef)
    {
        foreach ($this->tempTableTypes as $type) {
            list ($tempTableDb, $tempTableName) = explode('.', $tempTableDef[$type]);
            DB::statement("DROP TABLE IF EXISTS `{$tempTableDb}`.`{$tempTableName}`");
        }
    }

    /**
     * Get a list of table names (not necessarily created ones!) for temporary tables
     * 
     * @param string[] $tableNames
     * @return [
     *     string $tableName => [
     *         string $type => string $tempTableOfType,
     *         ...
     *     ],
     *     ...
     * ]
     */
    public function listTemporaryTableNames($tableNames)
    {
        $result = [];
        
        foreach ($tableNames as $tableName) {
            foreach ($this->tempTableTypes as $type) {
                $tempTableOfType = $this->prefixTableForTransaction($tableName, $type);
                $result[$tableName][$type] = $tempTableOfType;
            }
        }
        
        return $result;
    }
    
    /**
     * Get ON DUPLICATE KEY UPDATE column listing for table
     * 
     * @param string $table
     * @return string
     */
    public function getTableColumns($table)
    {
        if (!isset($this->columnStore[$table])) {
            $this->columnStore[$table] = $this->prefixedCache("columns.{$table}", function () use ($table) {
                return DB::getSchemaBuilder()->getColumnListing($table);
            });
        }
        
        return $this->columnStore[$table];
    }
    
    /**
     * Cache for MergeManager-related computations
     * 
     * @param string $subject
     * @param callable $computation
     * @return mixed
     */
    public function prefixedCache(string $subject, callable $computation)
    {
        $cacheKey = "{$this->cachePrefix}.{$subject}";
        return Cache::rememberForever($cacheKey, $computation);
    }
    
    /**
     * Prefixes table name for transaction table
     * 
     * @param string $tableName
     * @param string $type
     * @return array [string|null $implodeEnumeration, string|null $autoKey]
     */
    private function prefixTableForTransaction(string $tableName, string $type): string
    {
        $database = $this->sliceManager->getSchema();
        $uid = $this->sliceManager->getUID();
        $ret = "{$tableName}_{$type}_{$uid}";
        
        // limit of table name length
        if (strlen($ret) > self::INDENTIFIER_MAX_LENGTH) {
            return substr($ret, 0, self::INDENTIFIER_MAX_LENGTH);
        }
        
        return "{$database}.{$ret}";
    }
}
