<?php

namespace InSegment\ApiCore\Services\SliceMerger;

use InSegment\ApiCore\Exceptions\ApiTransactorException;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Arr;

class MergeOperation
{
    /**
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeManager  
     */
    protected $mergeManager;
    
    /**
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeOptions
     */
    protected $options;
    
    /**
     * @var \InSegment\ApiCore\Services\SliceMerger\CopyRecords
     */
    protected $copyRecords;
    
    /**
     * Slice manager
     *
     * @var \InSegment\ApiCore\Interfaces\SliceManagementInterface
     */
    protected $sliceManager;

    /**
     * Show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable 
     */
    protected $showCreateParser;
    
    /**
     * @var array
     */
    protected $movableTableTypes;
    
    /**
     * @var \InSegment\ApiCore\Models\SupplementaryTable
     */
    protected $autoKeyRemapModel;
    
    /**
     * @var \InSegment\ApiCore\Models\SupplementaryTable
     */
    protected $generationsMatchModel;
    
    /**
     * @var \InSegment\ApiCore\Models\SupplementaryTable
     */
    protected $uuidDeletesModel = null;
    
    /**
     * @var \InSegment\ApiCore\Models\SupplementaryTable|null
     */
    protected $realDeletesModel = null;
    
    /**
     * @var \InSegment\ApiCore\Models\SupplementaryTable|null
     */
    protected $logsModel = null;
    
    /**
     * @var string
     */
    protected $uid;
    
    /**
     * @var array [
     *     string $dependentTable => array [
     *          string $table => array $keys,
     *          ...
     *     ],
     *     ...
     * ]
     */
    protected $dependentTablesRelatedKeysMap;
    
    /**
     * @var string|null
     */
    private $currentTable = null;
    
    /**
     * @var string|null
     */
    private $currentOp = 'initial';
    
    /**
     * @var array [
     *    string $table => bool,
     *    ...
     * ]
     */
    private $processedTables = [];
    
    /**
     * @var array [
     *     string $table => int $amountAffected,
     *     ...
     * ]
     */
    private $affectedByTable = [];
    
    /**
     * @var array [
     *     string $table => string $keyName,
     *     ...
     * ]
     */
    private $groupsForDelete = [];
    
    /**
     * @var bool
     */
    private $started = false;
    
    /**
     * Constructor
     * 
     * @param \InSegment\ApiCore\Services\SliceMerger\MergeManager $mergeManager
     * @param \InSegment\ApiCore\Services\SliceMerger\MergeOptions $options
     */
    public function __construct(MergeManager $mergeManager, MergeOptions $options)
    {
        $this->mergeManager = $mergeManager;
        $this->options = $options;
        
        $this->sliceManager = $mergeManager->getSliceManager();
        $this->showCreateParser = $mergeManager->getShowCreateParser();
        $this->movableTableTypes = $mergeManager->getMovableTableTypes();
        
        $this->uid = $this->sliceManager->getUID();
        
        $this->setSupplementaryModels();
        $this->setDependentTablesRelatedKeysMap();
        
        $this->copyRecords = new CopyRecords($this);
    }
    
    public function getMergeManager(): MergeManager
    {
        return $this->mergeManager;
    }
    
    /**
     * @return \Illuminate\Database\Eloquent\Model
     */
    public function getGenerationsMatchModel()
    {
        return $this->generationsMatchModel;
    }
    
    /**
     * @return \Illuminate\Database\Eloquent\Model
     */
    public function getAutoKeyRemapModel()
    {
        return $this->autoKeyRemapModel;
    }
    
    /**
     * @param callable $postProcessing
     */
    public function perform(callable $postProcessing)
    {
        try {
            $this->prepareAndResetOperation();
            $this->autoKeyRemapModel->disposeUnused();
            $this->setGroupsForDelete();
            DB::transaction(function () use ($postProcessing) {
                $this->mergeTransaction($postProcessing);
            });
            $this->started = false;
        } catch (\Throwable $exception) {
            if ($this->currentOp === 'postProcess') {
                throw (new ApiTransactorException(ApiTransactorException::CODE_POST_PROCESS_FAILED, $exception))->compile();
            } else {
                throw (new ApiTransactorException(ApiTransactorException::CODE_FAIL_UNDONE, $exception))
                    ->compile($this->currentOp, $this->currentTable);
            }
        }
    }
    
    /**
     * Delete records from insertion by query and everything related to its UUID
     * 
     * @param \Illuminate\Database\Query\Builder $query
     * @param string $tableToRemoveFrom
     * @param array $typesToDelete
     */
    public function removeFromPendingInsertion(Builder $query, string $tableToRemoveFrom, ...$typesToDelete): array
    {
        $this->prepareAndResetOperation();
        $this->establishUuidDeletesTable();
        
        $hasMeetTheTable = false;
        
        foreach (array_keys($this->options->tempTables) as $table) {
            $this->setCurrentTable($table);
            
            // The array of tempTables is sorted in order from dependencies to dependent tables
            // So, if we delete from the table at some place in that list, no tables depending on that table
            // should be in the previous entries of that array.
            // 
            // If the array of tempTables is sorted incorrectly (as it is sorted by the developer, not automatically),
            // the merge will abort with exception in mergeTransaction.
            // 
            // So, skip all tables prioir to that one which needs records to be deleted from
            
            if ($table === $tableToRemoveFrom) {
                $hasMeetTheTable = true;
                $this->makeUuidDeletesByQuery($query, $tableToRemoveFrom, $typesToDelete);
            }
            
            if ($hasMeetTheTable) {
                $this->deleteRelatedToTableDeletedDependencies($table, $typesToDelete);
            }
            
            $this->processedTables[$table] = true;
        }
        
        if (!$hasMeetTheTable) {
            throw new \Exception("The table of deletion is not in the slices's temporary tables");
        }
        
        $this->disbandUuidDeletesTable();
        $this->started = false;
        return $this->affectedByTable;
    }
    
    /**
     * @param \Illuminate\Database\Query\Builder $query
     * @param string $realTable
     * @param array $typesToDelete
     * @return int
     */
    protected function makeUuidDeletesByQuery(Builder $query, string $realTable, array $typesToDelete): int
    {
        $autoKey = $this->showCreateParser->getAutoKey($realTable);
        $affected = 0;
        
        if ($autoKey !== null) {
            $tempTable = $this->options->tempTables[$realTable];

            foreach ($typesToDelete as $type) {
                $affected += $this->uuidDeletesModel->insertFromQuery($query, "{$tempTable[$type]}.{$autoKey}");
            }
            
            if ($affected === 0) {
                return $affected;
            }
        }
        
        $affected = $query->delete();
        $this->affectedByTable[$realTable] += $affected;
        
        return $affected;
    }
    
    /**
     * The generations table contains map of UUID to some table, and real id in the real table,
     * which is, initially, equals to NULL, until the table is merged.
     *    
     * As we cannot allocate auto-increment primary keys in advance before we insert the rows,
     * we must call LAST_INSERT_ID() to retrieve the first inserted id. All ids of mass INSERT
     * are consecutive when innodb_autoinc_lock_mode = 1 (this should be configured on the db server)
     * and so we can determine auto-increment key of N'th inserted as LAST_INSERT_ID() + (N - 1)
     *    
     * As we cannot properly relate the rows in advance not knowing their future ids, the order of insertion
     * must be precomputed in uuidables.json, and we must insert first the rows, which do not need any
     * not inserted relations, and which are needed by relations to be inserted later
     * 
     * @param callable $postProcessing
     */
    protected function mergeTransaction(callable $postProcessing)
    {
        foreach (array_keys($this->options->tempTables) as $table) {
            $this->setCurrentTable($table);
            $this->updateIdentifiersOfTableInsertedDependencies($table);
            $this->deleteRowsBeforeMerge($table);
            $this->mergeTemporaryTablesToRealOne($table);
            $this->deleteRowsAfterMerge($table);
            $this->processedTables[$table] = true;
        }
        
        $this->currentTable = null;
        $this->updateLogsTableIdentifiers();
        $this->currentOp = 'postProcess';
        DB::statement("SET FOREIGN_KEY_CHECKS=1");
        $postProcessing($this->sliceManager);
        DB::statement("SET FOREIGN_KEY_CHECKS=0");
    }
    
    /**
     * The dependencies of the records in the table, for which fields of the table hold temporary identifiers,
     * must be inserted before the table records can be inserted. After that, in temporary tables, intentifiers
     * must be updated to the new real identifiers of the records in the table it depends on
     * 
     * @param string $table
     */
    protected function updateIdentifiersOfTableInsertedDependencies(string $table)
    {
        $this->currentOp = 'updateIdentifiers';
        $fieldsRelatedToAffectedDependencies = $this->getFieldsRelatedToAffectedDependencies($table);
        
        // If there was a result of determination, update exactly one time each needed key
        if (count($fieldsRelatedToAffectedDependencies)) {
            foreach (array_keys($fieldsRelatedToAffectedDependencies) as $dependentKey) {
                $tempTable = $this->options->tempTables[$table];
                
                foreach ($this->movableTableTypes as $type) {
                    $this->updateRelationsInTemp($tempTable[$type], $dependentKey);
                }
            }
        }
    }
    
    /**
     * Delete records in the table, which have dependencies for which fields of the table hold temporary identifiers
     * 
     * @param string $table
     * @param array $typesToDelete
     */
    protected function deleteRelatedToTableDeletedDependencies(string $table, array $typesToDelete)
    {
        $this->currentOp = 'deleteRelatedToDependencies';
        $fieldsRelatedToAffectedDependencies = $this->getFieldsRelatedToAffectedDependencies($table);
        
        // If there was a result of determination, update exactly one time each needed key
        if (count($fieldsRelatedToAffectedDependencies)) {
            foreach (array_keys($fieldsRelatedToAffectedDependencies) as $dependentKey) {
                $tempTable = $this->options->tempTables[$table];
                
                foreach ($typesToDelete as $type) {
                    $this->deleteRelationsInTemp($table, $tempTable[$type], $dependentKey);
                }
            }
        }
    }
    
    /**
     * @param string $table
     * @return array [
     *     string $key => bool,
     *     ...
     * ]
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    protected function getFieldsRelatedToAffectedDependencies(string $table): array
    {
        $insertedDependentsRelatedFields = [];

        // Tables which are necessary for that table to be merged
        $mapCurrentTableDependsOn = $this->dependentTablesRelatedKeysMap[$table] ?? [];

        // Iterate over every table that table depends on
        foreach ($mapCurrentTableDependsOn as $dependencyTable => $dependencyTableKeys) {
            // If that table was not yet merged, that means UUID'ables are in wrong order
            // (order of UUID'ables is defined by the developer, not by API library)
            if (!isset($this->processedTables[$dependencyTable])) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_UUIDABLES_WRONG_ORDER))
                    ->compile('`' . collect($dependencyTableKeys)->implode('`, `') . '`', $table, $dependencyTable);
            }

            // Determine, whether there was any insert into any table this table is dependent on
            // If there was no inserts, then the theoretically dependent key in practice only
            // depends on the records that were already present in DB
            if ($this->affectedByTable[$dependencyTable] > 0) {
                foreach ($dependencyTableKeys as $key) {
                    $insertedDependentsRelatedFields[$key] = true;
                }
            }
        }
        
        return $insertedDependentsRelatedFields;
    }
    
    /**
     * Perform deletions from the table before merging to the table if there were any deletions registered
     * 
     * @param string $table
     */
    protected function deleteRowsBeforeMerge(string $table): void
    {
        $this->currentOp = 'deleteBeforeMerge';
        $deletesTable = $this->getDeletesTable();
        
        if ($deletesTable && isset($this->groupsForDelete[$table])) {
            $this->performTableDeletions($deletesTable, $table, $this->groupsForDelete[$table]);
        }
    }
    
    /**
     * Merge temporary tables into real ones
     * Merge insert, update and other tables into real table
     * 
     * @param string $table
     */
    protected function mergeTemporaryTablesToRealOne(string $table): void
    {
        $tempTable = $this->options->tempTables[$table];
        
        foreach ($this->movableTableTypes as $type) {
            $this->currentOp = 'moveType' . ucfirst($type);
            $affected = $this->copyRecords->moveRecordsFromTempToRealTable($table, $tempTable, $type);

            if (in_array($type, [MergeManager::TEMP_TABLE_TYPE_INSERT, MergeManager::TEMP_TABLE_TYPE_INSERT_IGNORE])) {
                $this->affectedByTable[$table] += $affected;
            }
        }
    }
    
    /**
     * Perform deletions from the table after merging to the table if there were any deletions
     * later in transaction for identifiers inserted to the table
     * 
     * @param string $table
     */
    protected function deleteRowsAfterMerge(string $table): void
    {
        $this->currentOp = 'deleteAfterMerge';
        $deletesTable = $this->getDeletesTable();
        
        // If some of the inserted records were deleted later in transaction
        if ($deletesTable && isset($this->groupsForDelete[$table]) && $this->affectedByTable[$table]) {
            // First update deletions table and have count of changes
            if (($updatedDeletions = $this->updateDeletionsInTemp($deletesTable, $table)) > 0) {
                // If there was any changed deletions, repeat the query to delete from real table
                $this->performTableDeletions($deletesTable, $table, $this->groupsForDelete[$table]);
            }
        }
    }
    
    /**
     * Update logs table so it can be used in post-processing to determine rows affected by transaction
     */
    protected function updateLogsTableIdentifiers(): void
    {
        $logsTable = $this->getLogsTable();
        
        if ($logsTable) {
            $this->currentOp = 'updateLogsTableIdentifiers';
            $this->currentTable = $logsTable;
            $this->updateRelationsInTemp($logsTable, 'key', true);
        }
    }
    
    /**
     * Delete rows from $table which keys are present in deletions table
     * 
     * @param string $deletionsTable
     * @param string $table
     * @param string $tableKeyForDeletion
     * @return int
     */
    protected function performTableDeletions(string $deletionsTable, string $table, string $tableKeyForDeletion)
    {
        return DB
            ::table($table)
            ->join("{$deletionsTable} as deletes", function ($join) use ($table, $tableKeyForDeletion) {
                return $join->where('deletes.table', '=', $table)->on('deletes.key', '=', "{$table}.{$tableKeyForDeletion}");
            })
            ->delete();
    }
    
    protected function setDependentTablesRelatedKeysMap(): void
    {
        $map = [];
        foreach ($this->options->uuidableMentions as $table => $dependents) {
            if (isset($this->options->tempTables[$table])) {
                foreach ($dependents as $dependentTable => $dependentKeys) {
                    $dependentKeysArray = Arr::wrap($dependentKeys);

                    if (isset($map[$dependentTable][$table])) {
                        $map[$dependentTable][$table] = array_unique(array_merge($map[$dependentTable][$table], $dependentKeysArray));
                    } else {
                        $map[$dependentTable][$table] = $dependentKeysArray;
                    }
                }
            }
        }
        
        $this->dependentTablesRelatedKeysMap = $map;
    }
    
    protected function setSupplementaryModels(): void
    {
        $autoKeyRemapClass = $this->options->autoKeyRemapClass;
        $generationsMatchClass = $this->options->generationsMatchClass;
        $uuidDeletesModelClass = $this->options->uuidDeletesModelClass;
        $realDeletesModelClass = $this->options->realDeletesModelClass;
        $logsModelClass = $this->options->logsModelClass;

        $this->autoKeyRemapModel = new $autoKeyRemapClass([], $this->uid);
        $this->generationsMatchModel = new $generationsMatchClass();
        $this->uuidDeletesModel = new $uuidDeletesModelClass([], $this->uid);
        
        if ($realDeletesModelClass) {
            $this->realDeletesModel = new $realDeletesModelClass([], $this->uid);
        }
        
        if ($logsModelClass) {
            $this->logsModel = new $logsModelClass([], $this->uid);
        }
    }
    
    /**
     * @return void
     */
    protected function setGroupsForDelete(): void
    {
        $deletesTable = $this->getDeletesTable();
        
        if ($deletesTable && count($this->options->deletionEnabledTables)) {
            $this->currentOp = 'groupsForDelete';
            $this->currentTable = $deletesTable;
            $deletesModelClass = $this->options->realDeletesModelClass;
            $this->groupsForDelete = $deletesModelClass::groupBy('table')->pluck('key_name', 'table')->toArray();
        } else {
            $this->groupsForDelete = [];
        }
    }
    
    protected function getDeletesTable(): ?string
    {
        return $this->realDeletesModel ? $this->realDeletesModel->getTable() : null;
    }
    
    protected function getLogsTable(): ?string
    {
        return $this->logsModel ? $this->logsModel->getTable() : null;
    }
    
    /**
     * Update UUID's in the temporary table relating to auto-keys of some real tables through remapper
     * 
     * @param string $tempTable
     * @param string $key
     * @param bool $isStringKey
     * @return int
     */
    private function updateRelationsInTemp(string $tempTable, string $key, bool $isStringKey = false)
    {
        $uuidField = $this->autoKeyRemapModel->uuidFieldName();
        $autoKeyField = $this->autoKeyRemapModel->autoKeyFieldName();
        $aliasForTempTable = 'temp_table';

        if ($isStringKey) {
            $joinKey = $this->castFieldStringAsUnsigned("`{$aliasForTempTable}`.`{$key}`");
        } else {
            $joinKey = "{$aliasForTempTable}.{$key}";
        }

        return DB
            ::table("{$tempTable} as {$aliasForTempTable}")
            ->join("{$this->autoKeyRemapModel->getTable()} as gens_table", "gens_table.{$uuidField}", '=', $joinKey)
            ->update(["{$aliasForTempTable}.{$key}" => DB::raw("`gens_table`.`{$autoKeyField}`")]);
    }
    
    /**
     * Delete records in the temporary table relating to auto-keys of some real tables through remapper
     * 
     * @param string $realTable
     * @param string $tempTable
     * @param string $key
     * @param bool $isStringKey
     * @return int
     */
    private function deleteRelationsInTemp(string $realTable, string $tempTable, string $key, bool $isStringKey = false)
    {
        $uuidField = $this->uuidDeletesModel->uuidFieldName();
        $aliasForTempTable = 'temp_table';

        if ($isStringKey) {
            $joinKey = $this->castFieldStringAsUnsigned("`{$aliasForTempTable}`.`{$key}`");
        } else {
            $joinKey = "{$aliasForTempTable}.{$key}";
        }
        
        $query = DB
            ::table("{$tempTable} as {$aliasForTempTable}")
            ->join("{$this->uuidDeletesModel->getTable()} as dels_table", "dels_table.{$uuidField}", '=', $joinKey);
            
        $autoKey = $this->showCreateParser->getAutoKey($realTable);
        if ($autoKey !== null) {
            $affected = $this->uuidDeletesModel->insertThroghTemp($query, "{$aliasForTempTable}.{$autoKey}");
            
            if ($affected === 0) {
                return $affected;
            }
        }
        
        $affected = $query->delete();
        $this->affectedByTable[$realTable] += $affected;
        
        return $affected;
    }
    
    /**
     * Update UUID's in the temporary table deleting remapped auto-keys of some real tables
     * 
     * @param string $deletionsTable
     * @param string $table
     * @return int
     */
    private function updateDeletionsInTemp(string $deletionsTable, string $table)
    {
        $uuidField = $this->autoKeyRemapModel->uuidFieldName();
        $autoKeyField = $this->autoKeyRemapModel->autoKeyFieldName();
        $joinKey = $this->castFieldStringAsUnsigned('`dels_table`.`key`');

        return DB
            ::table("{$deletionsTable} as dels_table")
            ->where('dels_table.table', '=', $table)
            ->join("{$this->autoKeyRemapModel->getTable()} as gens_table", "gens_table.{$uuidField}", '=', $joinKey)
            ->update(["dels_table.key" => DB::raw("`gens_table`.`{$autoKeyField}`")]);
    }

    /**
     * Cast string field as unsigned to fix mariaDB issue:
     * "In other cases, arguments are compared as floating point, or real, numbers."
     * @see https://mariadb.com/kb/en/library/type-conversion/#rules-for-conversion-on-comparison
     *
     * @param string $field
     * @return \Illuminate\Database\Query\Expression
     */
    private function castFieldStringAsUnsigned(string $field)
    {
        return DB::raw("CAST({$field} AS UNSIGNED)");
    }
    
    private function setCurrentTable(string $table): void
    {
        $this->currentTable = $table;
        
        if ($table !== null && !isset($this->affectedByTable[$table])) {
            $this->affectedByTable[$table] = 0;
        }
    }
    
    /**
     * To be able to iterate over dependencies, for example, for deletion the entities and related data before merge,
     * we need to be able to reset the state of MergeOperation
     * 
     * @throws \Exception
     */
    private function prepareAndResetOperation(): void
    {
        if ($this->started) {
            throw new \Exception("Merge operation has already started");
        }
        
        $this->started = true;
        $this->currentOp = 'reset';
        $this->currentTable = null;
        $this->processedTables = [];
        $this->affectedByTable = [];
    }
    
    private function establishUuidDeletesTable(): void
    {
        $uuidDeletesModelClass = $this->options->uuidDeletesModelClass;
        $uuidDeletesModelClass::setDefaultOptions([
            'inMemory' => false,
            'temporary' => true,
            'uid' => $this->uid,
        ]);
        
        $uuidDeletesModelClass::establish([]);
    }
    
    private function disbandUuidDeletesTable(): void
    {
        $uuidDeletesModelClass = $this->options->uuidDeletesModelClass;
        $uuidDeletesModelClass::disband();
    }
}
