<?php

namespace InSegment\ApiCore\Services;

use InSegment\ApiCore\Models\SliceRemap;
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Models\GenerationsMatch;

use InSegment\ApiCore\Services\SliceMerger\MergeManager;
use InSegment\ApiCore\Services\SliceMerger\MergeOptions;
use InSegment\ApiCore\Exceptions\ApiTransactorException;

use Illuminate\Support\Facades\DB;

abstract class SliceOperationBase
{
    /**
     * Tables which are copied
     *
     * @var string[]
     */
    protected $copyTables = [];
    
    /**
     * Tables which are targeted as id sources
     * 
     * @var string[]
     */
    protected $targetTables = [];
    
    /**
     * Slice Operation
     *
     * @var \InSegment\ApiCore\Models\SliceOperation
     */
    protected $slice;
    
    /**
     * Slice Merge manager
     *
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeManager 
     */
    protected $mergeManager;

    /**
     * Slice Merge operation
     *
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeOperation|null
     */
    private $mergeOperation;
    
    /**
     * Tables to the keys which are mentioned in some tables and thus may have updated key
     *
     * @var array [
     *     string $tableName => string|string[] $keys,
     *     ...
     * ]
     */
    protected $uuidableMentions = [];
    
    /**
     * Tables to the dependent tables with their dependent keys to be related to that table
     *
     * @var array [
     *     string $tableName => [
     *          string $dependentTable => string[] $dependentKeys
     *     ],
     *     ...
     * ]
     */
    protected $uuidableMergeMentions = [];
    
    /**
     * Tables to temporary tables
     *
     * @var array [
     *     string $tableName => [
     *         string $type => string $tempTableOfType,
     *         ...
     *     ],
     *     ...
     * ] 
     */
    protected $tempTables = [];
    
    /**
     * Counters to check estimates against those for affections when inserting records
     *
     * @var array [
     *     'new' => [
     *          string $table => int $amount,
     *          ...
     *     ],
     *     'written' => [
     *          string $table => int $amount,
     *          ...
     *     ]
     * ] 
     */
    protected $estimateCounters = [];
    
    /**
     * Class name of auto key remapper
     *
     * @var string 
     */
    protected $autoKeyRemapClass = SliceRemap::class;
    
    /**
     * Merge slice tables into real ones
     * 
     * @param callable $onMerge
     * @param callable $postProcessing
     * @return bool
     */
    public function finishCopy(callable $onMerge, callable $postProcessing)
    {
        $this->slice->setCounters($this->estimateCounters);
        try {
            DB::transaction(function () use ($onMerge, $postProcessing) {
                $onMerge($this->slice);
                $this->getMergeOperation()->perform($postProcessing);
            });
        } catch (ApiTransactorException $exception) {
            $this->slice->status = SliceOperation::STATUS_ABORT;
            $this->slice->save();
            throw $exception;
        }

        foreach ($this->tempTables as $tempTableDef) {
            $this->mergeManager->dropTempTable($tempTableDef);
        }

        $autoKeyRemapClass = $this->autoKeyRemapClass;
        $autoKeyRemapClass::disband($this->slice->getUID());
        $this->slice->status = SliceOperation::STATUS_DONE;
        return $this->slice->save();
    }
    
    /**
     * Get name of temporary table for model class
     * 
     * @param string $modelClass
     * @return string|null
     */
    public function getTempInsertTable(string $modelClass)
    {
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        return $this->tempTables[$modelTable][MergeManager::TEMP_TABLE_TYPE_INSERT] ?? false;
    }
    
    /**
     * Get name of temporary updates table for model class
     * 
     * @param string $modelClass
     * @return string|null
     */
    public function getTempUpdateTable(string $modelClass)
    {
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        return $this->tempTables[$modelTable][MergeManager::TEMP_TABLE_TYPE_UPDATE] ?? false;
    }
    
    /**
     * Get name of temporary changes table for model class
     * 
     * @param string $modelClass
     * @return string|null
     */
    public function getTempChangesTable(string $modelClass)
    {
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        return $this->tempTables[$modelTable][MergeManager::TEMP_TABLE_TYPE_CHANGES] ?? false;
    }
    
    /**
     * Delete records from insertion by query and everything related to its UUID
     * This works recursively, and relations of deleted UUID's are also deleted
     * However, relations made by other means (i.e. not to auto-increment primary key)
     * cannot be deleted in that manner
     * 
     * @param \Illuminate\Database\Eloquent\Builder
     *        |\Illuminate\Database\Query\Builder
     *        |\Illuminate\Database\Eloquent\Relations\Relation $query
     * @param string $table
     * @return array [
     *     string $table => int $amountDeleted,
     *     ...
     * ]
     */
    public function removeEntitiesInsertion($query, string $table = null): array
    {
        if ($query instanceof \Illuminate\Database\Query\Builder) {
            $baseQuery = (clone $query);
            $table = $table ?? $baseQuery->from;
        } else if ($query instanceof \Illuminate\Database\Eloquent\Builder) {
            $baseQuery = (clone $query)->toBase();
            $table = $table ?? $query->getModel()->getTable();
        } else if ($query instanceof \Illuminate\Database\Eloquent\Relations\Relation) {
            $baseQuery = (clone $query)->getBaseQuery();
            $table = $table ?? $query->getQuery()->getModel()->getTable();
        } else {
            throw new \Exception('Unsupported query type');
        }
        
        $insertType = MergeManager::TEMP_TABLE_TYPE_INSERT;
        $deletesByTable = $this->getMergeOperation()->removeFromPendingInsertion($query, $table, $insertType);
        foreach ($deletesByTable as $table => $amount) {
            $negativeAmount = 0 - $amount;
            $this->slice->increaseTableCounter('new', $table, $negativeAmount);
        }
        
        return $deletesByTable;
    }
    
    private function getMergeOperation(): SliceMerger\MergeOperation
    {
        if ($this->mergeOperation) {
            return $this->mergeOperation;
        }
        
        // TODO: consider to use options builder
        $mergeOptions = new MergeOptions();
        $mergeOptions->tempTables = $this->tempTables;
        $mergeOptions->uuidableMentions = $this->uuidableMergeMentions;
        $mergeOptions->autoKeyRemapClass = $this->autoKeyRemapClass;
        $mergeOptions->generationsMatchClass = GenerationsMatch::class;
        return $this->mergeOperation = $this->mergeManager->createOperation($mergeOptions);
    }
    
}
