<?php

namespace InSegment\ApiCore\Services;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

use Illuminate\Support\Str;
use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Exceptions\ApiTransactorException;
use InSegment\ApiCore\Scopes\TransactionSearchScope;
use InSegment\ApiCore\Models\DeletedTransactionRecord;
use InSegment\ApiCore\Models\SupplementarySliceTable;
use InSegment\ApiCore\Models\LogReceivedRecord;
use InSegment\ApiCore\Models\UUIDGeneration;
use InSegment\ApiCore\Models\UUIDReserve;
use InSegment\ApiCore\Models\SliceRemap;
use InSegment\ApiCore\Models\GenerationsMatch;
use InSegment\ApiCore\Models\TransactionData;
use InSegment\ApiCore\Services\MicroEvent;
use InSegment\ApiCore\Services\ExceptionDebug;
use InSegment\ApiCore\Services\SliceMerger\MergeManager;
use InSegment\ApiCore\Services\SliceMerger\MergeOptions;
use InSegment\ApiCore\Api\DataTransferObject;

use InSegment\ApiCore\Interfaces\SliceManagementInterface;
use InSegment\ApiCore\Traits\SliceManagementTrait;
use InSegment\ApiCore\Traits\SliceSchemaTrait;

use Illuminate\Database\Eloquent\Model;

/**
 * Class Transactor
 *
 * @package App
 */
final class Transactor implements SliceManagementInterface
{
    use SliceManagementTrait;
    use SliceSchemaTrait;
    
    /**
     * An instance
     * 
     * @var \InSegment\ApiCore\Services\Transactor 
     */
    private static $instance = null;
    
    /**
     * Builder for marks of deletion of records
     *
     * @var \InSegment\ApiCore\Services\BufferedBuilder
     */
    private $deletedMarkingsQuery = null;
    
    /**
     * Builder for received records insertUpdate
     *
     * @var \InSegment\ApiCore\Services\BufferedBuilder
     */
    private $recievedLogQuery = null;
    
    /**
     * Builder for UUID insertUpdate
     *
     * @var \InSegment\ApiCore\Services\BufferedBuilder
     */
    private $uuidLogQuery = null;
    
    /**
     * A Transaction
     * 
     * @var \InSegment\ApiCore\Models\Transaction
     */
    private $transaction;
    
    /**
     * An state of Transaction model
     * 
     * @var \InSegment\ApiCore\Models\TransactionData
     */
    private $transactionData;
    
    /**
     * An array holding all created temporary tables
     * 
     * @var array 
     */
    private $temporaryTables = [];
    
    /**
     * Show create table parser
     * 
     * @var \InSegment\ApiCore\Services\ParseCreateTable 
     */
    private $showCreateParser;
    
    /**
     * Uuidables mentions merge
     * 
     * @var array 
     */
    private $uuidablesMentions;
    
    /**
     * An array of ids for existence in update table for each table for table type
     * 
     * @var array 
     */
    private $existence = ['insert' => [], 'update' => []];
    
    /**
     * Session identifier
     * 
     * @var string
     */
    private $sessionId;
    
    /**
     * Uuidables
     * 
     * @var array 
     */
    private $uuidables;
    
    /**
     * Reserved UUIDs by table
     * 
     * @var \InSegment\ApiCore\Models\UUIDReserve
     */
    private $uuidReserve;
    
    /**
     * BufferedWriter instance
     * 
     * @var \InSegment\ApiCore\Services\BufferedWriter
     */
    private $bufferedWriter;
    
    /**
     * Results of static inspection of rules
     * 
     * @var array 
     */
    private $inspectionResults;
    
    /**
     * Slice Merge manager
     *
     * @var \InSegment\ApiCore\Services\SliceMerger\MergeManager 
     */
    private $mergeManager;
    
    /**
     * Allowed transaction tables
     * 
     * @var array
     */
    private static $allowedTransactionTables;
    
    /**
     * Allowed transaction tables key names
     * 
     * @var array
     */
    private static $allowedTransactionTablesKeys;
    
    /**
     * Tables which enabled deletion from them
     * 
     * @var array 
     */
    private static $deletionEnabledTables;
    
    /**
     * If Transactor is currently active
     * 
     * @var bool
     */
    private static $isWatching = false;

    /**
     * This does not make Transactor fully independent between uses in same process.
     * But at least this clears major bugs if called between uses.
     */
    public static function clearStatic(): void {
        self::$instance = null;
        self::$allowedTransactionTables = null;
        self::$allowedTransactionTablesKeys = null;
        self::$deletionEnabledTables = null;
        self::$isWatching = false;
    }
    
    /**
     * Only those tables are available to open transactions for
     * 
     * @param bool $readOnly
     * @param string $rootClass
     * @return array
     */
    public static function getAllowedTransactionTables(bool $readOnly = false, string $rootClass = null) : array
    {
        $workLoad = ['ew-with-read-only' => $readOnly];

        if (!isset($rootClass)) {
            if (!isset(self::$allowedTransactionTables)) {
                self::getInstance()->runInspections($workLoad);
            }
            
            return self::$allowedTransactionTables;
        }
        
        return DTODefs::instance()->performInspection($rootClass, 'enabled-write', $workLoad)['enabled'];
    }
    
    /**
     * Only those tables keys are available to open transactions for
     * 
     * @return array
     */
    public static function getAllowedTransactionTablesKeys() : array
    {
        if (!isset(self::$allowedTransactionTablesKeys)) {
            $workLoad = ['ew-with-read-only' => false];
            self::getInstance()->runInspections($workLoad);
        }
        
        return self::$allowedTransactionTablesKeys;
    }
    
    /**
     * Only those tables which enabled deletion from them during transact
     * 
     * @param bool $readOnly
     * @param string $rootClass
     * @return array
     */
    public static function getDeletionEnabledTables(bool $readOnly = false, string $rootClass = null) : array
    {
        $workLoad = ['ew-with-read-only' => $readOnly];

        if (!isset($rootClass)) {
            if (isset(self::$deletionEnabledTables)) {
                return self::$deletionEnabledTables;
            }

            self::getInstance()->runInspections($workLoad);
            return self::$deletionEnabledTables;
        }
        
        return DTODefs::instance()->performInspection($rootClass, 'enabled-write', $workLoad)['deletable'];
    }

    /**
     * Get transaction UID
     * 
     * @return string
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function getUID(): string
    {
        if (!$this->transactionData->hasUID()) {
            self::requireTransactionState();
            throw (new ApiTransactorException(ApiTransactorException::CODE_UNABLE_TO_GENERATE_UID))->compile();
        }
        
        return $this->transactionData->getUID();
    }
    
    /**
     * Get schema name for transaction tables
     * 
     * @return string
     */
    public static function getTransactionSchema()
    {
        return self::getInstance()->getSchema();
    }
    
    /**
     * Lists column unique data
     * 
     * @param string $rootClass
     * @return array
     */
    public static function getColumnUniqueData(string $rootClass): array
    {
        $instance = self::getInstance();
        $uniqueData = [];
        
        foreach (self::getAllowedTransactionTables(true, $rootClass) as $modelClass => $table) {
            $table = (new $modelClass)->getTable();
            
            $external = DTODefs::getExternalKeyDefaultName($modelClass, true);
            $uniqueData[$table] = $instance->showCreateParser->getColumnUniqueData($table);
            isset($external) && array_unshift($uniqueData[$table], ["external:{$external}"]);
        }
        
        return $uniqueData;
    }
    
    /**
     * Get unique keys which are using specified column(s) of the specified model
     * 
     * @param string $modelClass
     * @param array $columns
     * @return array
     */
    public static function getUniquesForColumns(string $modelClass, array $columns): array
    {
        $instance = self::getInstance();
        $table = (new $modelClass)->getTable();
        $results = [];
        
        foreach ($instance->showCreateParser->getColumnUniqueData($table) as $unique) {
            if (count(array_intersect($columns, $unique))) {
                $results[] = $unique;
            }
        }
        
        return $results;
    }
    
    /**
     * Get enabled eager loads list
     * 
     * @param string $rootClass
     * @param array $categories
     * @return array
     */
    public static function getEnabledEagers(string $rootClass, array $categories = ['all', 'export']): array
    {
        $instance = self::getInstance();
        !isset($instance->inspectionResults['eagers']) && $instance->runInspections([], $rootClass);
        $inspection = $instance->inspectionResults['eagers'];
        
        if (self::isWatching()) {
            // nullify by reference logic
            foreach ($inspection['tables'] as $table => &$relationPathes) {
                if (!isset($instance->temporaryTables[$table])) {
                    foreach ($relationPathes as &$path) {
                        $path = null;
                    }
                }
            }
        }
        
        $eagers = array_values(array_unique(array_filter(
            (in_array('all', $categories) ? Arr::get($inspection, 'relations.all', []) : []) +
            (in_array('export', $categories) ? Arr::get($inspection, 'relations.export', []) : [])
        )));
        
        return $eagers;
    }
    
    /**
     * Get preload Closure for $class, if any
     *
     * @param string $parentClass
     * @param string $modelClass
     * @param string $classInQuestion
     * @param string $extKeysImplode
     * @return \Closure|null
     */
    public static function getPreloads(string $parentClass, string $classInQuestion, string $extKeysImplode)
    {
        $instance = self::getInstance();
        !isset($instance->inspectionResults['preload']) && $instance->runInspections();
        return $instance->inspectionResults['preload']['suppliers'][$parentClass][$classInQuestion][$extKeysImplode] ?? null;
    }
    
    /**
     * Use input data to fulfill wildcards for preload
     * 
     * @param string $parentClass
     * @param array $data
     * @return null
     */
    public static function fulfilReferenceWildcards(string $parentClass, &$data)
    {
        $instance = self::getInstance();
        !isset($instance->inspectionResults['preload']) && $instance->runInspections();
        
        foreach (Arr::get($instance->inspectionResults, "preload.referenceWildcards.{$parentClass}", []) as $class => $wildcards) {
            // For same key there may be different rules in the same rules file that specify a wildcard
            // Instead of only collecting by the latest wildcard rule, should collect by every of them, or this produces a hard-to-catch bug
            foreach ($wildcards as $externalKey => $wildcardsForKey) {
                foreach ($wildcardsForKey as $wildcard) {
                    $array = is_array($wildcard);
                    $count = $array ? count($wildcard) : 1;

                    if ($count > 1) {
                        $segments = [];
                        foreach ($wildcard as $k => $wildCardPart) {
                            $segments[] = array_values(array_filter(data_get($data, $wildCardPart, [])));
                        }

                        // Add results to existing results if some were put by the previous wildcard
                        $out = $instance->inspectionResults['preload']['reference'][$parentClass][$class][$externalKey] ?? [];
                        foreach ($segments as &$segment) {
                            foreach ($segment as $key => &$value) {
                                $out[$key][] = &$value;
                            }
                        }
                        
                        $instance->inspectionResults['preload']['reference'][$parentClass][$class][$externalKey] = $out;
                    } else {
                        if ($array) {
                            $wildcard = $wildcard[0];
                        }

                        // Merge results of this wildcard with results of previous wildcards.
                        $instance->inspectionResults['preload']['reference'][$parentClass][$class][$externalKey] = array_unique(array_merge(
                            array_values(array_filter(data_get($data, $wildcard, []))),
                            $instance->inspectionResults['preload']['reference'][$parentClass][$class][$externalKey],
                        ));
                    }
                }
            }
        }
    }
    
    /**
     * Method to be called by global exception handler to transform any exceptions happened during active transaction
     * 
     * @param \Throwable $exception
     * @return \Throwable
     */
    public static function affectOnYourWatch(\Throwable $exception): \Throwable
    {
        if (!self::$isWatching || self::isNotPrepared() || $exception instanceof ApiTransactorException) {
            return $exception;
        }
        
        if (!config('api_core.transaction_enable_debug_abnormal')) {
            self::getInstance()->transaction->abortTransaction(false);
        }
        
        return new ApiTransactorException(ApiTransactorException::CODE_SERVER_FAULT, $exception);
    }
    
    /**
     * Checks if the Transactor instance is prepared correctly
     * 
     * @return bool
     */
    public static function isPrepared()
    {
        return self::getInstance()->transaction->isActive();
    }
    
    /**
     * Checks if the Transactor instance is not prepared at all
     * 
     * @return bool
     */
    public static function isNotPrepared()
    {
        return self::getInstance()->transaction->isNotPrepared();
    }

    /**
     * Get status of Transactor
     * 
     * @param array|null $columns
     * @return array
     */
    public static function status(array $columns = null)
    {
        try {
            $errMode = ['errmode' => DB::connection()->getPdo()->getAttribute(\PDO::ATTR_ERRMODE)];
        } catch (\Throwable $exception) {
            $errMode = ['errmode' => 'unknown'];
            ExceptionDebug::makeDebug($errMode, $exception, 'errmode_exception');
        }
        
        $instance = self::getInstance();
        $source = $instance->transactionData->getData() + $instance->transaction->toArray() + $errMode;
        
        return isset($columns) ? array_intersect_key($source, array_flip($columns)) : $source;
    }
    
    /**
     * If the Transactor transacting $table and temporary ones exist, returns true
     * 
     * @param string $modelClass A string representing Model class
     * @return boolean
     */
    public static function doesHaveTables(string $modelClass)
    {
        $instance = self::getInstance();
        $allowedTransactionTables = self::getAllowedTransactionTables();
        
        if (isset($allowedTransactionTables[$modelClass])) {
            $modelTable = $allowedTransactionTables[$modelClass];
            return isset($instance->temporaryTables[$modelTable]);
        }
        
        return false;
    }
    
    /**
     * If the Transactor transacting $table, returns temporary insert table
     * Otherwise returns null
     * 
     * @param string $modelClass A string representing Model class
     * @param bool $strict
     * @return string|null
     */
    public static function getInsertTable(string $modelClass, bool $strict = false)
    {
        $instance = self::getInstance();
        $allowedTransactionTables = self::getAllowedTransactionTables();
        
        if (isset($allowedTransactionTables[$modelClass])) {
            $modelTable = $allowedTransactionTables[$modelClass];
            if (isset($instance->temporaryTables[$modelTable])) {
                return $instance->temporaryTables[$modelTable]['insert'];
            }
        }
        
        if ($strict) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile((new $modelClass)->getTable());
        }
        
        return null;
    }
    
    /**
     * Execute actions on an instance of Transactor within the state special handling for Eloquent operations on Models
     * if the transaction is prepared, or just in plain state otherwise
     * 
     * @param callable $transactionAction
     * @return mixed
     */
    public static function watchingTables(callable $transactionAction = null)
    {
        if (self::$isWatching) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_RECURSIVE_TRANSACTION))->compile();
        }
        
        // do not check foreign keys while importing
        DB::statement("SET FOREIGN_KEY_CHECKS=0");

        self::startWatchingTables();

        try {
            $result = $transactionAction(self::getInstance());
        } catch (\Throwable $e) {
            throw $e;
        } finally {
            self::stopWatchingTables();

            // re-enable foreign key checks
            DB::statement("SET FOREIGN_KEY_CHECKS=1");
            self::$isWatching = false;
        }

        return $result;
    }
    
    /**
     * Whether the Transactor is in the mode of special handling for Eloquent operations on Models
     * 
     * @return bool
     */
    public static function isWatching()
    {
        return self::$isWatching;
    }
    
    /**
     * Enable special handling for Eloquent operations on Models
     * 
     * @return null
     */
    private static function startWatchingTables()
    {
        JoinResolver::$enabled = true;
        
        if (self::isPrepared()) {
            self::$isWatching = true;
            
            foreach (array_keys(self::getAllowedTransactionTables()) as $class) {
                TransactionSearchScope::applyTo($class);
            }

            $instance = self::getInstance();
            $writer = $instance->getBufferedWriter();
            BufferedBuilder::$buffer = $writer;
            MicroEvent::registerEvent(BufferedWriter::class, 'flushed', function ($flushedWriter) use ($instance, $writer) {
                if (self::$isWatching && $flushedWriter->getId() === $writer->getId()) {
                    $instance->transactionData->saveData();
                }
            });
        }
    }
    
    /**
     * Disable special handling for Eloquent operations on Models
     * 
     * @return null
     */
    private static function stopWatchingTables()
    {
        if (self::isPrepared()) {
            BufferedBuilder::$buffer = null;
            TransactionSearchScope::disable();
            self::$isWatching = false;
        }
        
        JoinResolver::$enabled = false;
    }
    
    /**
     * Require to be prepared and not abnormal
     */
    public static function requireTransactionState()
    {
        if (self::isNotPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_SEQUENCE))->compile('not prepared');
        } else if (!self::isPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_FAIL_NOT_UNDONE))->compile();
        }
    }
    
    /**
     * Get the (cached) key existence of the Model instance with specified id in $type table for Model of $modelClass
     * 
     * @param string $modelClass
     * @param mixed $keyValue
     * @param string $keyName
     * @return bool
     */
    public static function getExistence(string $modelClass, $keyValue, string $keyName = 'id')
    {
        $instance = self::getInstance();
        $allowedTransactionTables = self::getAllowedTransactionTables();
        
        if (isset($allowedTransactionTables[$modelClass])) {
            $modelTable = $allowedTransactionTables[$modelClass];
            
            if (isset($instance->temporaryTables[$modelTable])) {
                if ((new $modelClass)->_incrementing) {
                    foreach (['insert', 'update'] as $type) {
                        if (!isset($instance->existence[$type][$modelTable])) {
                            self::addToExistence($instance, $type, $modelTable, $keyName);
                        }
                        
                        if (isset($instance->existence[$type][$modelTable][$keyValue])) {
                            return true;
                        }
                    }
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get the list of IDs of the objects, that were changed during the current transaction.
     * 
     * @param string $modelClass
     * @param string $keyName
     * @return array|false
     */
    public static function getTransactedChangesIds(string $modelClass, string $keyName = 'id')
    {
        $instance = self::getInstance();
        if (!($modelTable = $instance->getModelTable($modelClass))) {
            return false;
        }
        
        $result = [];

        foreach (['insert', 'update'] as $type) {
            if (!isset($instance->existence[$type][$modelTable])) {
                self::addToExistence($instance, $type, $modelTable, $keyName);
            }

            $result = array_merge($result, array_keys($instance->existence[$type][$modelTable]));
        }

        return array_unique($result);
    }
    
    /**
     * Scope of the all transacted models (including those not changed directly)
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $modelClass
     * @param string $keyName
     * @param bool $strict
     * @return \Illuminate\Database\Eloquent\Builder|null
     */
    public static function receivedModelsScope($query, string $keyName = null, bool $strict = false)
    {
        $model = $query->getModel();
        if (!isset($keyName)) {
            $keyName = $model->getKeyName();
        }
        
        if (($receivedIdsQuery = self::getInstance()->getReceivedIdsQuery(get_class($model), $keyName))) {
            return $query
                ->whereRaw("`{$keyName}` IN ({$receivedIdsQuery->toSql()})")
                ->mergeBindings($receivedIdsQuery->toBase());
        } else if ($strict) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile($model->getTable());
        } else {
            return null;
        }
    }

    /**
     * Get or create Transaction
     * 
     * @param mixed $data Counts.
     * @return void
     */
    public static function acquire($data)
    {
        $rootClass = data_get($data, 'root', '');
        $rootDTOs = DTODefs::getGlobalRootDTOs();

        if (!$rootClass || !in_array($rootClass, $rootDTOs)) {
            $rootClass = $rootDTOs[0];
        }

        self::prepare($rootClass, data_get($data, 'count', []));
    }

    /**
     * Set or update Object
     * 
     * @param string $type
     * @param mixed $data
     * @param bool $update
     * @return mixed
     */
    public static function import($type, $data, $update)
    {
        $result = self::watchingTables(function (Transactor $transactor) use ($type, $data, $update) {
            $ret = (new DataTransferObject($type, $data, Str::singular($type)))->processIncoming($update);
            $transactor->getBufferedWriter()->flushAll();
            $transactor->transactionData->saveData();
            return $ret;
        });
            
        return $result;
    }
    
    /**
     * Undo Transaction
     * 
     * @return null
     */
    public static function undoTransaction()
    {
        self::watchingTables(function (Transactor $transactor) {
            return $transactor->undo();
        });
    }
    
    /**
     * Close Transaction
     * 
     * @param bool $clear
     * @return mixed
     */
    public static function close(&$clear)
    {
        $result = self::watchingTables(function (Transactor $transactor) use ($clear) {
            $mergeOutput = DTODefs::onTransactionMerge($transactor);
            
            return $transactor->commit(function (Transactor $transactor) use ($mergeOutput) {
                return DTODefs::postProcessTransaction($transactor, $mergeOutput);
            }, $clear);
        });
        
        return $result;
    }

    /**
     * Get query for IDs of Models received in transaction (including not changed ones)
     * 
     * @param string $modelClass
     * @param string $keyName
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getReceivedIdsQuery(string $modelClass, string $keyName = null)
    {
        $instance = self::getInstance();
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        if (!isset($keyName)) {
            $keyName = $modelExample->getKeyName();
        }
        
        if (!isset($instance->temporaryTables[$modelTable])) {
            return false;
        }
        
        return LogReceivedRecord
            ::select(['key'])
            ->where('table', '=', $modelTable)
            ->where('key_name', '=', $keyName);
    }
    
    /**
     * Get query for IDs of Models received in transaction (including not changed ones)
     * 
     * @param string $modelClass
     * @param string $keyName
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getKeysForDeleteQuery(string $modelClass, string $keyName = null)
    {
        $instance = self::getInstance();
        $modelExample = (new $modelClass);
        $modelTable = $modelExample->getTable();
        if (!isset($keyName)) {
            $keyName = $modelExample->getKeyName();
        }
        
        if (!isset($instance->temporaryTables[$modelTable])) {
            return false;
        }
        
        return DeletedTransactionRecord
            ::select(['key'])
            ->where('table', '=', $modelTable)
            ->where('key_name', '=', $keyName);
    }
    
    /**
     * Get BufferedWriter instance
     * 
     * @return \InSegment\ApiCore\Services\BufferedWriter
     */
    public function getBufferedWriter()
    {
        return $this->bufferedWriter;
    }
    
    /**
     * Get table for model class
     * 
     * @param string $modelClass
     * @return string|boolean
     */
    protected function getModelTable(string $modelClass)
    {
        $allowedTransactionTables = self::getAllowedTransactionTables();

        if (!isset($allowedTransactionTables[$modelClass])) {
            return false;
        }

        $modelTable = $allowedTransactionTables[$modelClass];
        
        if (!isset($this->temporaryTables[$modelTable])) {
            return false;
        }

        return $modelTable;
    }

    /**
     * Populate Transactor->existence property with a new value.
     * 
     * @param \InSegment\ApiCore\Services\Transactor $instance
     * @param string $type
     * @param string $modelTable
     * @param string $keyName
     */
    protected static function addToExistence(Transactor $instance, string $type, string $modelTable, string $keyName)
    {
        if (!isset($instance->temporaryTables[$modelTable])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile($modelTable);
        }
        
        $instance->existence[$type][$modelTable] = array_flip(DB::table($instance->temporaryTables[$modelTable][$type])->pluck($keyName)->toArray());
    }
    
    /**
     * If the Transactor transacting $table, returns temporary update table
     * Otherwise returns null
     * 
     * @param string $modelClass A string representing Model class
     * @param bool $strict
     * @return string|null
     */
    public static function getUpdateTable(string $modelClass, bool $strict = false)
    {
        $instance = self::getInstance();
        $allowedTransactionTables = self::getAllowedTransactionTables();
        
        if (isset($allowedTransactionTables[$modelClass])) {
            $modelTable = $allowedTransactionTables[$modelClass];
            if (isset($instance->temporaryTables[$modelTable])) {
                return $instance->temporaryTables[$modelTable]['update'];
            }
        }
        
        if ($strict) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile((new $modelClass)->getTable());
        }
        
        return null;
    }
    
    /**
     * Get session object of the specified class
     * 
     * @param string $class
     * @return mixed
     */
    public static function getSessionObject(string $class) {
        return self::getInstance()->transactionData->getSessionObject($class);
    }
    
    /**
     * Get session object of the specified class
     * 
     * @param string $class
     * @param mixed $object
     * @return null
     */
    public static function setSessionObject(string $class, $object = null) {
        return self::getInstance()->transactionData->setSessionObject($class, $object);
    }
    
    /**
     * Attempt to commit the transaction effects into the actual db
     * 
     * @param \Closure|null $postProcessing
     * @param bool $clear
     * @return array result of commit (empty array on success)
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function commit(\Closure $postProcessing = null, bool $clear = true)
    {
        if (!self::isPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_SEQUENCE))->compile('not prepared');
        }
        
        /**
         * Check that all-right, and we received what we wanted in sufficient amount
         */
        foreach ($this->transactionData->getData('reserved') as $table => $reservedAmount) {
            $receivedAmount = $this->transactionData->getRecievedAmount($table);

            if ($receivedAmount < $reservedAmount && !$this->transactionData->getIsDynamicUnderweight($table)) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_WEIGHT))
                    ->compile('Transaction is underweight')->setData(['table' => $table]);
            } else if ($receivedAmount > $reservedAmount) {
                throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_WEIGHT))
                    ->compile('Transaction is overweight')->setData(['table' => $table]);
            }
        }
        
        return $this->doCommitTables($postProcessing, $clear);
    }
    
    /**
     * Attempt to undo the transaction effects
     * 
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public function undo()
    {
        if (self::isNotPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_SEQUENCE))->compile('not prepared');
        }
        
        $this->tryReverse();
        $this->transaction->abortTransaction(true);
        BufferedBuilder::$buffer = null;
        TransactionSearchScope::disable();
        self::$isWatching = false;
    }
    
    /**
     * This method notifies Transactor about new supplied objects
     * 
     * @param bool $wasDirty
     * @param bool $dontSave
     * @param \Illuminate\Database\Eloquent\Model $model A Model instance
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public static function receive(bool $wasDirty, bool $dontSave, $model)
    {
        $class = get_class($model);
        $writeTable = $model->getTable();
        $instance = self::getInstance();
        $table = $instance->getTableAllowedByClass($class);
        if (!isset($instance->temporaryTables[$table])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile($table);
        }
        
        $keyName = $model->getKeyName();
        $isIncrementing = (new $class)->_incrementing;
        $isNew = !$dontSave && $writeTable == $instance->temporaryTables[$table]['insert'];
        
        if ($isIncrementing && !$instance->transactionData->canReceiveIncrementing($table)) {
            $received = $instance->transactionData->getRecievedAmount($table);
            $limit = $instance->transactionData->getReservedAmount($table);
            
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_WEIGHT))
                ->compile('Excessive objects supplied')->setData(['table' => $table, 'received' => $received, 'limit' => $limit]);
        }
        
        $instance->transactionData->incrementReceived($table, $wasDirty, $isNew, $isIncrementing);
        
        if ($keyName && ($key = $model->getAttribute($keyName)) !== null) {
            // with plain Eloquent builder this won't work, but the builder is BufferedBuilder,
            // so these will convert into bulk INSERT ...ON DUPLICATE KEY UPDATE statements
            $instance->recievedLogQuery->insertUpdate(['table' => $table, 'key_name' => $keyName, 'key' => $key]);
            
            if ($isNew && $isIncrementing) {
                $instance->uuidLogQuery->insertUpdate(['uuid' => $key, 'used' => true]);
            }
        }
    }
    
    /**
     * Register a mark-for-deletion of the record for the table
     * 
     * @param string $table
     * @param string $keyName
     * @param int $keyVal
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    public static function markDeleted(string $table, $keyName, $keyVal)
    {
        $instance = self::getInstance();
        if (!isset($instance->temporaryTables[$table])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile($table);
        }
        
        $instance->deletedMarkingsQuery->insert(['table' => $table, 'key_name' => $keyName, 'key' => $keyVal]);
    }
    
    /**
     * This method gives a primary key for a new model
     * Do not increment the counter yet - what if 
     * 
     * @param \Illuminate\Database\Eloquent\Model|string $class A Model instance or a string representing its class
     */
    public static function nextId(string $class) {
        $instance = self::getInstance();
        $table = $instance->getTableAllowedByClass($class);
        
        if (!isset($instance->uuidReserve, $instance->temporaryTables[$table])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_TABLE_NOT_INCLUDED))->compile($table);
        }
        
        return $instance->uuidReserve->nextId($table);
    }
    
    /**
     * Prepare temporary tables
     * 
     * @param string $rootClass
     * @param array $tableCountData
     * @return null
     * @throws \App\Exceptions\ApiTransactorException
     */
    public static function prepare(string $rootClass, array $tableCountData)
    {
        if (self::isPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_SEQUENCE))->compile('not closed');
        } else if (!self::isNotPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_FAIL_NOT_UNDONE))->compile();
        }
        
        if (!is_a($test = new $rootClass, $rootClass) || !is_a($test, Model::class)) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_SERVER_FAULT))
                ->compile('Invalid argument: \$rootClass must be a valid class name of a class that extends Model')
                ->setData(['rootClass' => $rootClass]);
        }
        
        $instance = self::getInstance();
        $instance->transaction->prepareTransaction($instance->sessionId, $rootClass);
        
        if (!$instance->transaction->uid) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_UNABLE_TO_GENERATE_UID))->compile();
        }
        
        SupplementarySliceTable::$sliceManager = $instance;
        
        $inspectionWorkLoad = ['auto-count' => &$tableCountData];
        $inspection = DTODefs::performInspection($rootClass, ['auto-count', 'enabled-write'], $inspectionWorkLoad);
        $instance->transactionData->setDynamicUnderweightData(Arr::get($inspection, 'auto-count.dynUW', []));
        self::$allowedTransactionTables = &$inspection['enabled-write']['enabled'];
        self::$allowedTransactionTablesKeys = &$inspection['enabled-write']['key-name'];
        $memoryEngineEnabled = $instance->getMemoryEngineEnabled($tableCountData);
        
        LogReceivedRecord::establish([
            'tablesToKeys' => self::$allowedTransactionTablesKeys,
            'inMemory' => $memoryEngineEnabled[LogReceivedRecord::class]
        ]);
        
        self::$deletionEnabledTables = &$inspection['enabled-write']['deletable'];
        if (count(self::$deletionEnabledTables)) {
            DeletedTransactionRecord::establish([
                'tablesToKeys' => array_intersect_key(self::$allowedTransactionTablesKeys, self::$deletionEnabledTables),
                'inMemory' => $memoryEngineEnabled[DeletedTransactionRecord::class]
            ]);
        }
        
        $instance->uuidReserve = new UUIDReserve;
        $instance->deletedMarkingsQuery = (new DeletedTransactionRecord)->newQuery();
        $instance->recievedLogQuery = (new LogReceivedRecord)->newQuery();
        $instance->uuidLogQuery = (new UUIDGeneration)->newQuery();
        
        if (count($instance->uuidables)) {
            UUIDGeneration::establish([
                'tables' => array_keys(array_intersect_key(self::$allowedTransactionTablesKeys, $instance->uuidables)),
                'inMemory' => $memoryEngineEnabled[UUIDGeneration::class]
            ]);
            
            foreach (array_intersect_key($tableCountData, self::$allowedTransactionTablesKeys, $instance->uuidables) as $table => $count) {
                if ($count > 0) {
                    $instance->uuidReserve->allocate($table, $count);
                }
            }
        }
        
        foreach (self::$allowedTransactionTables as $modelClass => $table) {
            if ((new $modelClass)->_incrementing) {
                if (!isset($tableCountData[$table])) {
                    throw (new ApiTransactorException(ApiTransactorException::CODE_COUNT_MISSING))->compile($table);
                }

                $reserveCount = (int) $tableCountData[$table];

                if ($reserveCount == 0) {
                    continue;
                } else if ($reserveCount < 0) { // TODO: some maximum value to check?
                    throw (new ApiTransactorException(ApiTransactorException::CODE_COUNT_INVALID))->compile($table);
                }

                $instance->createTempTablesForModel($table, $memoryEngineEnabled[$table] ?? false, $reserveCount);
            } else {
                $instance->createTempTablesForModel($table, $memoryEngineEnabled[$table] ?? false);
            }
        }
        
        $instance->transactionData->saveData();
        $instance->transaction->activateTransaction();
    }
    
    /**
     * Maps tables to whether their counts are in memory limits
     * 
     * @param array $counts
     * @return array
     */
    private function getMemoryEngineEnabled($counts)
    {
        $sumCounts = array_sum($counts);
        $memoryEngineLimits = DTODefs::getTableMemoryLimits();
        
        if (!isset($memoryEngineLimits['*'])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_NO_DEFAULT_MEMORY_ENGINE_LIMIT))->compile();
        }
        
        $result = [];
        
        foreach (self::$allowedTransactionTables as $table) {
            if (isset($counts[$table])) {
                $result[$table] = ($memoryEngineLimits[$table] ?? $memoryEngineLimits['*']) > $counts[$table];
            } else {
                $result[$table] = false;
            }
        }
        
        foreach ([LogReceivedRecord::class, DeletedTransactionRecord::class, UUIDGeneration::class] as $supplementary) {
            $table = (new $supplementary)->getTable();
            $result[$supplementary] = ($memoryEngineLimits[$table] ?? $memoryEngineLimits['*']) > $sumCounts;
        }
        
        return $result;
    }
    
    /**
     * Clone the table structure into the new temporary tables for transaction,
     * reserve the specified amount of id's in the original table and
     * prepare the temporary tables for inserts to receive objects with these id's
     * 
     * @param string $modelTable A Model table
     * @param bool $allowInMemory
     * @param int|null $reserveCount
     * @return null
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    private function createTempTablesForModel(string $modelTable, bool $allowInMemory, int $reserveCount = null)
    {
        $isUuidable = DTODefs::isUUIDable($modelTable);
        $this->showCreateParser->loadColumnData($modelTable);
        
        $allowedTablesKeys = self::getAllowedTransactionTablesKeys();
        $uuidableMentions = $this->getUuidablesMentions($allowedTablesKeys);
        $uuidableTableMentions =
            ($uuidableTableMentions = Arr::get($uuidableMentions, $modelTable))
                ? array_combine($uuidableTableMentions, $uuidableTableMentions)
                : null;
        
        $tempTables = $this->mergeManager->createTempTable($modelTable, $uuidableTableMentions, $isUuidable, $allowInMemory);
        $this->temporaryTables[$modelTable] = $tempTables;
        $this->transactionData->setReservedAmount($modelTable, $reserveCount);
    }
    
    /**
     * Get merge of uuidables mention columns
     * 
     * @param array $allTransactionTblsToKeys
     * @return array
     */
    private function getUuidablesMentions($allTransactionTblsToKeys)
    {
        if (!isset($this->uuidablesMentions)) {
            $allUuidables = DTODefs::getUUIDables();
            
            $this->uuidablesMentions = array_map(
                function ($item) {
                    return array_unique(Arr::wrap($item));
                }, array_merge_recursive(
                    // get base tables to keys for those tables, which are uuidable and in transaction
                    array_intersect_key($allTransactionTblsToKeys, $allUuidables),
                    // get values like ['mentioned_table' => ['some_key', ...], 'mentioned_table2' => ['some_key', ...]]
                    // from those uuidable, which base tables are in transaction
                    ...array_values(array_intersect_key($allUuidables, $allTransactionTblsToKeys))
                )
            );
        }
        
        return $this->uuidablesMentions;
    }
    
    /**
     * Get tables from transactions of crashed or timed out sessions
     * 
     * @param \Illuminate\Database\Eloquent\Collection|array $clearableSessions
     * @param bool $old
     * @return array
     */
    public static function tablesForGC($clearableSessions, bool $old = false) {
        $database = self::getTransactionSchema();
        $tableNameField = "Tables_in_{$database}";

        $regex = self::transactionTablesRegex($old);
        $allTransactionTables = DB::select("SHOW TABLES FROM `{$database}` WHERE `{$tableNameField}` REGEXP '{$regex}'");
        $outdatedTransactionTables = [];
        
        foreach ($allTransactionTables as $record) {
            $table = $record->$tableNameField;
            
            // a table name may have incomplete session id due to table name length limit
            $tableSession = ltrim(strrchr($table, '_'), '_');
            
            foreach ($clearableSessions as $uid) {
                if (strpos($uid, $tableSession) === 0) {
                    $outdatedTransactionTables[$uid][] = $table;
                    break;
                }
            }
        }
        
        foreach ($clearableSessions as $uid) {
            $supplementaries = [
                last(explode('.', (new DeletedTransactionRecord([], $uid))->getTable())),
                last(explode('.', (new SliceRemap([], $uid))->getTable())),
                last(explode('.', (new UUIDGeneration([], $uid))->getTable())),
                last(explode('.', (new LogReceivedRecord([], $uid))->getTable()))
            ];
            
            foreach ($supplementaries as $table) {
                if (count(DB::select("SELECT * FROM `information_schema`.`tables` WHERE `table_schema` = '{$database}' and table_name = '{$table}'")) > 0) {
                    $outdatedTransactionTables[$uid][] = $table;
                }
            }
        }
        
        return $outdatedTransactionTables;
    }
    
    /**
     * Drops temporary tables with new data
     * 
     * @return null
     */
    private function tryReverse()
    {
        foreach ($this->temporaryTables as $tempTable) {
            foreach (['insert', 'update'] as $type) {
                try {
                    list($database, $tempTableName) = explode('.', $tempTable[$type]);
                    DB::statement("DROP TABLE IF EXISTS `{$database}`.`{$tempTableName}`");
                    unset($database, $tempTableName);
                } catch (\Throwable $ex) {}
            }
        }
        
        UUIDGeneration::disband();
        LogReceivedRecord::disband();
        DeletedTransactionRecord::disband();
    }
    
    /**
     * Actual transaction finishing
     * 
     * @param \Closure|null $postProcessing
     * @param bool $clear
     * @return array empty if everything gone OK
     */
    private function doCommitTables(\Closure $postProcessing = null, bool $clear = true): array
    {
        $failedToCommit = [];
        $formerStatus = $this->status();

        try {
            // $allowedTablesKeys = self::getAllowedTransactionTablesKeys();
            $allUuidables = DTODefs::getUUIDables();
            $sortedTempTables = array_intersect_key(array_merge($allUuidables, $this->temporaryTables), $this->temporaryTables);
            $uuidableMergeMentions = [];
            foreach ($allUuidables as $table => $dependents) {
                $mergeableDependents = array_intersect_key($dependents, $sortedTempTables);
                if (count($mergeableDependents)) {
                    $uuidableMergeMentions[$table] = $mergeableDependents;
                }
            }
            
            // TODO: consider to use options builder
            $mergeOptions = new MergeOptions();
            $mergeOptions->tempTables = $sortedTempTables;
            $mergeOptions->uuidableMentions = $uuidableMergeMentions;
            $mergeOptions->deletionEnabledTables = self::$deletionEnabledTables;
            $mergeOptions->autoKeyRemapClass = UUIDGeneration::class;
            $mergeOptions->generationsMatchClass = GenerationsMatch::class;
            $mergeOptions->realDeletesModelClass = DeletedTransactionRecord::class;
            $mergeOptions->logsModelClass = LogReceivedRecord::class;
            
            $postProcessingCallback = function (Transactor $instance) use ($postProcessing) {
                self::stopWatchingTables();
                return $postProcessing($instance);
            };
                
            $this->mergeManager->createOperation($mergeOptions)->perform($postProcessingCallback);
        } catch (ApiTransactorException $exception) {
            $this->transaction->abortTransaction(false);
            throw $exception;
        }
        
        if ($clear) {
            UUIDGeneration::disband();
            LogReceivedRecord::disband();
            DeletedTransactionRecord::disband();
            foreach ($this->temporaryTables as $tempTable) {
                $this->mergeManager->dropTempTable($tempTable);
            }
        }
        
        $this->transaction->finishTransaction($clear);

        try {
            $warning_messages = DB::select('SHOW WARNINGS');
            if (count($warning_messages)) {
                Arr::set($failedToCommit, "warning", $warning_messages);
            }
        } catch (\Throwable $exception) {
            Arr::set($failedToCommit, "warning", 'Failed to SHOW WARNINGS');
        }
        
        if (count($failedToCommit)) {
            $failedToCommit['status'] = $formerStatus;
        }

        return $failedToCommit;
    }
    
    /**
     * Run inspection for allowed tables
     * 
     * @param array $workLoad
     * @param string $rootClass
     * @return $this
     */
    private function runInspections(array $workLoad = [], string $rootClass = null)
    {
        $this->inspectionResults = DTODefs::performInspection(
            $rootClass ?? $this->transaction->root_class,
            ['preload', 'enabled-write', 'eagers'], $workLoad
        );
        
        self::$allowedTransactionTables = &$this->inspectionResults['enabled-write']['enabled'];
        self::$allowedTransactionTablesKeys = &$this->inspectionResults['enabled-write']['key-name'];
        self::$deletionEnabledTables = &$this->inspectionResults['enabled-write']['deletable'];
        
        return $this;
    }
    
    /**
     * Get regex to match any temporary table
     * 
     * @param bool $old
     * @return string
     */
    private static function transactionTablesRegex(bool $old = false): string
    {
        $mergeTables = array_reduce(
            DTODefs::getGlobalRootDTOs(),
            function ($carry, $rootClass) {
                return $carry + self::getAllowedTransactionTables(false, $rootClass);
            }, []
        );
            
        $tablesOptions = implode('|', $mergeTables);
        $typesOptions = implode('|', MergeManager::CLEARABLE_TEMP_TABLE_TYPES);
        $sessionIdChars = $old ? 'a-z0-9' : '0-9';
        return "({$tablesOptions})_({$typesOptions})_([{$sessionIdChars}]+)";
    }
    
    /**
     * Get table for a class
     * 
     * @param string $class
     * @return string
     * @throws \InSegment\ApiCore\Exceptions\ApiTransactorException
     */
    private function getTableAllowedByClass($class) : string
    {
        if (!self::isPrepared()) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_WRONG_SEQUENCE))->compile('not prepared');
        }
        
        $allowedTransactionModels = self::getAllowedTransactionTables();
        if (!isset($allowedTransactionModels[$class])) {
            throw (new ApiTransactorException(ApiTransactorException::CODE_UNEXPECTED_CLASS))->compile($class);
        }
        
        return $allowedTransactionModels[$class];
    }
    
    /**
     * A singleton, no instantiation from outside
     */
    private function __construct()
    {
        $this->uuidables = DTODefs::getUUIDables();
        $this->sessionId = strtolower(\Illuminate\Support\Facades\Session::getId());
        $this->showCreateParser = ParseCreateTable::getInstance();
        $this->mergeManager = new MergeManager($this, $this->showCreateParser);
        $this->bufferedWriter = new BufferedWriter();
        $this->transactionData = new TransactionData();
        $this->transaction = $this->transactionData->getTransaction();
        
        if ($this->transactionData->hasUID()) {
            $this->transactionData->loadData();
            $this->temporaryTables = $this->mergeManager->listTemporaryTableNames(array_keys($this->transactionData->getData('received')));
            
            SupplementarySliceTable::$sliceManager = $this;
            $this->uuidReserve = new UUIDReserve;
            $this->deletedMarkingsQuery = (new DeletedTransactionRecord)->newQuery();
            $this->recievedLogQuery = (new LogReceivedRecord)->newQuery();
            $this->uuidLogQuery = (new UUIDGeneration)->newQuery();
        }
    }
    
    /**
     * Get Transactor instance
     * 
     * @return \InSegment\ApiCore\Services\Transactor
     */
    private static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new Transactor();
        }
        
        return self::$instance;
    }
    
    /**
     * Get count of records was written for table
     * 
     * @param string $table
     * @return int
     */
    public function getWrittenCount(string $table): int
    {
        return $this->transactionData->getWrittenAmount($table);
    }
    
    /**
     * Get count of records was inserted into table
     * 
     * @param string $table
     * @return int
     */
    public function getNewCount(string $table): int
    {
        return $this->transactionData->getNewAmount($table);
    }

}
