<?php

namespace InSegment\ApiCore\Traits;

use InSegment\ApiCore\Services\ParseCreateTable;

use Illuminate\Database\Eloquent\Model;

trait BuildsDictionaries
{
    /**
     * Create table parser instance
     *
     * @var \InSegment\ApiCore\Services\ParseCreateTable
     */
    protected $parseCreateTable;
    
    /**
     * Uniques for classes
     *
     * @var array [
     *     string $class1 => [...],
     *     string $class1 => [...],
     *     ...
     * ]
     */
    protected $uniques;
    
    /**
     * Closures from keys
     *
     * @var \Closure[]
     */
    protected static $closures = [];
    
    /**
     * Get all rules enumeration or a set rules for specific class
     * If there is no rule for class, and the default value is not present, the method should throw \Exception
     * 
     * @param string|null $class
     * @param mixed|null $default
     * @throws \Exception
     * @return array
     */
    public abstract function getRules(string $class = null, $default = null);
    
    /**
     * Get default external key name for the specified class
     * If the class does not exist in known rules, an \Exception will be thrown
     * 
     * @param string $class
     * @param bool $info
     * @throws \Exception
     * @return string|null
     */
    public function getExternalKeyDefaultName($class, bool $info = false)
    {
        $rulesForClass = $this->getRules($class);
        
        if (is_array($rulesForClass)) {
            if (isset($rulesForClass['id'])) {
                $externalKeyName = $rulesForClass['id'];

                if ($info) {
                    return "{$externalKeyName} as {id: key}";
                }

                return $externalKeyName;
            }
            
            return null;
        } else {
            return $rulesForClass;
        }
    }
    
    /**
     * Get default input key name for the specified class
     * If the class does not exist in known rules, an \Exception will be thrown
     * 
     * @param string $class
     * @return string|null
     */
    public function getInputKeyDefaultName($class)
    {
        $rulesForClass = $this->getRules($class);
        
        if (is_array($rulesForClass)) {
            if (isset($rulesForClass['id'])) {
                return 'id';
            }
            
            return null;
        } else {
            return $rulesForClass;
        }
    }
    
    /**
     * Get value from data for external key. If the key is not specified, the default one will be taken
     * If the default key could not be determined because the known rules for class do not exist, an \Exception will be thrown
     * 
     * @param string $class
     * @param mixed $data
     * @param string|null $key
     * @throws \Exception
     * @return mixed
     */
    public function getExtKeyByData(string $class, $data, string $key = null)
    {
        if ($key === null) {
            $key = $this->getInputKeyDefaultName($class);
        }
        
        if (is_array($data)) {
            if (isset($data[$key])) {
                return $data[$key];
            } else {
                $debug = json_encode([
                    'given_data' => $data,
                    'expected_key' => $key
                ], JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR);

                throw new \Exception(
                    "Object of class {$class} cannot be processed, because the necessary"
                    . " key is not found in the input data. Debug: {$debug}"
                );
            }
        } else {
            return $data;
        }
    }
    
    /**
     * Get key values from model to be used as keys in result for model
     * 
     * @param \Illuminate\Database\Eloquent\Model $model
     * @param array $extKey
     * @return array
     * @throws \Exception
     */
    public function getResultKeyValues(Model $model, array $extKey)
    {
        switch ($extKey['type']) {
            case 'keys':
                $indexOut = [];

                foreach ($extKey['value'] as $attributeOfClass) {
                    $indexOut[] = $model->getAttribute($attributeOfClass);
                }

                return $indexOut;
            case 'callable':
                $stringCallable = $extKey['value'];
                $closure = static::$closures[$stringCallable] ?? (static::$closures[$stringCallable] = \Closure::fromCallable(explode('@', $stringCallable)));
                return $closure($model);
            case 'method': return $model->{$extKey['value']}();
            default: throw new \Exception("Unknown ext key type: '{$extKey['type']}'");
        }
    }
    
    /**
     * Get info about specified external keys of class from input
     * An external key may be array, string or null
     * If the 
     * 
     * @param string $class
     * @param array|string|null $extKey
     * @param array|string|null $input
     * @param array $param [
     *     'multiple' => bool,
     *     'index' => bool
     * ]
     * @return array
     */
    public function getExtKeyInfo(string $class, $extKey, $input, array $param = [])
    {
        $keyOut = [];
        $multiple = $param['multiple'] ?? false;
        $extIsArray = is_array($extKey);
        
        // if asked, check the specified external key
        !empty($param['check']) && $this->checkIsValidExternalKey($class, $extKey);
        
        // if we need to build index, prepare an empty array
        if (($index = ($multiple && !empty($param['index'])))) {
            $indexOut = [];
        }
        
        // no key, or many and no data
        if (!isset($extKey) || ($multiple && (!is_array($input) || !count($input)))) {
            return $index ? [$keyOut, $indexOut] : $keyOut;
        }
        // many data or many keys
        else if ($multiple || $extIsArray) {
            $extKeyEnum = $extIsArray ? $extKey : [$extKey];
            // input data iteration, if there is no multiple data, iterate an array with it, to not repeat the code
            foreach (($multiple ? $input : [$input ?? []]) as $dataKey => $dataValue) {
                // the output for the item, i.e ['foo' => 'bar', 'baz' => 123], where 'foo' and 'baz' are attributes
                // of the class in question, 'bar' and 123 are input values for these attributes extracted from the input
                $dataOut = [];

                // if we need an index, we should prevent the leak of variables from the previous interation,
                // so reset the pointer of index array to its root each time we have next input
                if ($index) {
                    $dataIndexOut = &$indexOut;
                    $dataIndexAttrValue = null;
                    $valueSet = false;
                }

                foreach ($extKeyEnum as $inputExtKeyPart => $attributeOfClass) {
                    if (is_int($inputExtKeyPart)) {
                        $inputExtKeyPart = $attributeOfClass;
                    }

                    // if we have input, extract the part of external key from the input
                    if (isset($input)) {
                        $dataOut[$attributeOfClass] = $attrValue = $this->getExtKeyByData($class, $dataValue, $extIsArray ? $inputExtKeyPart : $extKey);
                    } else {
                        $dataOut[$attributeOfClass] = $attrValue = null;
                    }

                    // if we don't need index, that's all; otherwise continue to build index
                    if (!$index) { continue; }

                    // build index, use reference logic to avoid recursion, each time we have new part
                    // we must have a boolean to store the setness of $dataIndexAttrValue, because the value itself can be null
                    if ($valueSet) {
                        // if we have a value set, and data by the index of that value is not an array, we have a key there
                        // so we need to point to an array first, forgetting the input key (we know it anyway and would set again)
                        if (!is_array($dataIndexOut[$dataIndexAttrValue])) {
                            $dataIndexOut[$dataIndexAttrValue] = [];
                        }

                        // now when the data by the index of the value is certainly an array, point to it
                        // it is also, still referenced in the root array, which is our index tree
                        $dataIndexOut = &$dataIndexOut[$dataIndexAttrValue];
                    }

                    // remember the value - we will need it should the next iteraion occur, also set the boolean that
                    // we have value, as when the $dataIndexAttrValue === null, we can't check we dont have it with isset
                    $dataIndexAttrValue = $attrValue;
                    $valueSet = true;
                    
                    // the only way there is array_key_exists, because the $dataIndexAttrValue can be null
                    // and $dataIndexAttrValue = null should be handled gracefully
                    if (!array_key_exists($dataIndexAttrValue, $dataIndexOut)) {
                        $dataIndexOut[$dataIndexAttrValue] = $dataKey;
                    }
                }

                // if we were asked to build keys for multiple data inputs, put in the result by the data key
                if ($multiple) {
                    $keyOut[$dataKey] = $dataOut;
                }
                // we know for certain that otherwise this is the the only iteration and the output is only one entry
                else {
                    $keyOut = $dataOut;
                }
            }
        // one data, one key
        } else {
            $value = isset($input) ? $this->getExtKeyByData($class, $input, $extKey) : null;
            $keyOut = [$extKey => $value];
            if ($index) { $indexOut[] = $value; }
        }
        
        return $index ? [$keyOut, $indexOut] : $keyOut;
    }

    /**
     * Check that specified external key is really an unique key in the table of the class
     * 
     * @param string $class
     * @throws \Exception
     * @param array|string|null $extKey
     */
    protected function checkIsValidExternalKey(string $class, $extKey)
    {
        if (!isset($this->uniques[$class])) {
            $this->uniques[$class] = $this->buildUniquesForClass($class);
        }
        
        $uniques = $this->uniques[$class];
        $navigate = is_array($extKey) ? $extKey : [$extKey];
        foreach ($navigate as $attributeOfClass) {
            if (!isset($uniques[$attributeOfClass])) {
                $debug = json_encode([
                    'class' => $class,
                    'expected_key' => $extKey,
                    'failed_column' => $attributeOfClass
                ], JSON_PRETTY_PRINT|JSON_PARTIAL_OUTPUT_ON_ERROR);

                throw new \Exception(
                    "Object of class {$class} does not have unique key in the table matching the columns"
                    . " of the specified external key or the order of columns is different. Debug: {$debug}"
                );
            }

            $uniques = $uniques[$attributeOfClass];
        }
    }
    
    /**
     * Build indexed unique result for class
     * 
     * @param string $class
     * @throws \Exception
     * @return array
     */
    protected function buildUniquesForClass(string $class)
    {
        if (!isset($this->parseCreateTable)) {
            $this->parseCreateTable = ParseCreateTable::getInstance();
        }
        
        $unuques = $this->parseCreateTable->getColumnUniqueData($this->getClassTable($class));
        
        // now build indexed result for uniques
        $result = [];
        foreach ($unuques as $uniqueKey) {
            $keyResult = &$result;
            $keyResultColumn = null;
            foreach ($uniqueKey as $column) {
                // build index, use reference logic to avoid recursion, each time we have new part
                if (isset($keyResultColumn)) {
                    if (!is_array($keyResult[$keyResultColumn])) {
                        $keyResult[$keyResultColumn] = [];
                    }

                    $keyResult = &$keyResult[$keyResultColumn];
                }

                $keyResultColumn = $column;
                $keyResult[$keyResultColumn] = true;
            }
        }
        
        return $result;
    }
    
    /**
     * Get list of unique keys for the specified class, handle possible failures with explanation
     * 
     * @param string $class
     * @throws \Exception
     * @return array
     */
    protected function getClassTable(string $class)
    {
        try {
            $example = new $class;
        } catch (\Exception $e) {
            throw new \Exception("Cannot instantiate the class '{$class}' to get unique columns from its table!", 0, $e);
        }
        
        if (method_exists($example, 'getTable')) {
            try {
                return $example->getTable();
            } catch (\Exception $e) {
                throw new \Exception("Error executing method '{$class}::getTable' to get unique columns from its table!", 0, $e);
            }
        } else {
            throw new \Exception("Method '{$class}::getTable' does not exists! Cannot determine unique columns for the class!");
        }
    }
    
}
