<?php

namespace InSegment\ApiCore\Traits;

use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Model;
use InSegment\ApiCore\Models\AttributeCache;
use InSegment\ApiCore\Services\AttributeCacheStore;

/**
 * To be used with \Illuminate\Database\Eloquent\Model
 */
trait CachesAttributes
{
    /**
     * Cache key, which could be changed (usable for API transactions)
     *
     * @var string
     */
    private $attributeCacheKey = '';
    
    /**
     * Fast access to an instance of attribute cache store
     *
     * @var \InSegment\ApiCore\Services\AttributeCacheStore|null
     */
    private static $attributeCacheStore;
    
    /**
     * Get store of cached characteristics of attributes
     * 
     * @return \InSegment\ApiCore\Models\AttributeCache
     */
    private function getAttributeCache()
    {
        $store = self::$attributeCacheStore ?? (self::$attributeCacheStore = AttributeCacheStore::getInstance());
        $attributeCache = $store->boxes[static::class][$this->attributeCacheKey] ?? $store->newStore(static::class, $this->attributeCacheKey);

        if ($attributeCache->fresh) {
            $attributeCache->casts = parent::getCasts();
            $attributeCache->table = parent::getTable();
            $attributeCache->fresh = false;
        }
        
        return $attributeCache;
    }
    
    /**
     * Get store of cached characteristics of attributes filled for $key
     * 
     * @param string $key
     * @return \InSegment\ApiCore\Models\AttributeCache
     */
    private function getFilledAttributeCache(string $key)
    {
        $attributeCache = $this->getAttributeCache();
        
        if (!isset($attributeCache->hasCast[$key])) {
            $attributeCache->hasCast[$key] = (bool) $this->hasCast($key);
            $attributeCache->hasMethod[$key] = (bool) method_exists($this, $key);
            $attributeCache->baseHasMethod[$key] = (bool) method_exists(Model::class, $key);
            $attributeCache->hasGetMutator[$key] = (bool) $this->hasGetMutator($key);
            $attributeCache->hasSetMutator[$key] = (bool) $this->hasSetMutator($key);
            $attributeCache->isDateAttribute[$key] = (bool) $this->isDateAttribute($key);
            $attributeCache->isJsonCastable[$key] = (bool) $this->isJsonCastable($key);
            $attributeCache->fillJson[$key] = (bool) Str::contains($key, '->');
            $attributeCache->inDates[$key] = (bool) in_array($key, $this->getDates());
            $attributeCache->cacheAttribute($key);
        }
        
        return $attributeCache;
    }
    
    /**
     * Get an attribute from the model.
     *
     * @param  string  $key
     * @return mixed
     */
    public function getAttribute($key)
    {
        if (! $key) {
            return;
        }
        
        $attributeCache = $this->getFilledAttributeCache($key);
        
        // If the attribute exists in the attribute array or has a "get" mutator we will
        // get the attribute's value. Otherwise, we will proceed as if the developers
        // are asking for a relationship's value. This covers both types of values.
        if (array_key_exists($key, $this->attributes) || $attributeCache->hasGetMutator[$key]) {
            return $this->getAttributeValue($key);
        }

        // Here we will determine if the model base class itself contains this given key
        // since we do not want to treat any of those methods are relationships since
        // they are all intended as helper methods and none of these are relations.
        if ($attributeCache->baseHasMethod[$key]) {
            return;
        }

        return $this->getRelationValue($key);
    }

    /**
     * Get a plain attribute (not a relationship).
     *
     * @param  string  $key
     * @return mixed
     */
    public function getAttributeValue($key)
    {
        $attributeCache = $this->getFilledAttributeCache($key);
        $value = $this->getAttributeFromArray($key);

        // If the attribute has a get mutator, we will call that then return what
        // it returns as the value, which is useful for transforming values on
        // retrieval from the model to a form that is more useful for usage.
        if ($attributeCache->hasGetMutator[$key]) {
            return $this->mutateAttribute($key, $value);
        }

        // If the attribute exists within the cast array, we will convert it to
        // an appropriate native PHP type dependant upon the associated value
        // given with the key in the pair. Dayle made this comment line up.
        if ($attributeCache->hasCast[$key]) {
            return $this->castAttribute($key, $value);
        }

        // If the attribute is listed as a date, we will convert it to a DateTime
        // instance on retrieval, which makes it quite convenient to work with
        // date fields without having to create a mutator for each property.
        if ($attributeCache->inDates[$key] && $value !== null) {
            return $this->asDateTime($value);
        }

        return $value;
    }

    /**
     * Get a relationship.
     *
     * @param  string  $key
     * @return mixed
     */
    public function getRelationValue($key)
    {
        $attributeCache = $this->getFilledAttributeCache($key);
        
        // If the key already exists in the relationships array, it just means the
        // relationship has already been loaded, so we'll just return it out of
        // here because there is no need to query within the relations twice.
        if ($this->relationLoaded($key)) {
            return $this->relations[$key];
        }

        // If the "attribute" exists as a method on the model, we will just assume
        // it is a relationship and will load and return results from the query
        // and hydrate the relationship's value on the "relationships" array.
        if ($attributeCache->hasMethod[$key]) {
            return $this->getRelationshipFromMethod($key);
        }
    }

    /**
     * Set a given attribute on the model.
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return $this
     */
    public function setAttribute($key, $value)
    {
        $attributeCache = $this->getFilledAttributeCache($key);
        
        // First we will check for the presence of a mutator for the set operation
        // which simply lets the developers tweak the attribute as it is set on
        // the model, such as "json_encoding" an listing of data for storage.
        if ($attributeCache->hasSetMutator[$key]) {
            return $this->{'set'.Str::studly($key).'Attribute'}($value);
        }

        // If an attribute is listed as a "date", we'll convert it from a DateTime
        // instance into a form proper for storage on the database tables using
        // the connection grammar's date format. We will auto set the values.
        elseif ($attributeCache->isDateAttribute[$key] && $value) {
            $value = $this->fromDateTime($value);
        }

        if ($attributeCache->isJsonCastable[$key] && $value !== null) {
            $value = $this->castAttributeAsJson($key, $value);
        }

        // If this attribute contains a JSON ->, we'll set the proper value in the
        // attribute's underlying array. This takes care of properly nesting an
        // attribute in the array's value in the case of deeply nested items.
        if ($attributeCache->fillJson[$key]) {
            return $this->fillJsonAttribute($key, $value);
        }

        $this->attributes[$key] = $value;

        return $this;
    }
    
    /**
     * Set the value of the "created at" attribute.
     *
     * @param  mixed  $value
     * @return $this
     */
    public function setCreatedAt($value)
    {
        $this->setAttribute(static::CREATED_AT, $value);

        return $this;
    }

    /**
     * Set the value of the "updated at" attribute.
     *
     * @param  mixed  $value
     * @return $this
     */
    public function setUpdatedAt($value)
    {
        $this->setAttribute(static::UPDATED_AT, $value);

        return $this;
    }

    /**
     * Determine if the given attribute exists.
     *
     * @param  mixed  $offset
     * @return bool
     */
    public function offsetExists($offset): bool
    {
        return $this->getAttribute($offset) !== null;
    }

    /**
     * Get the value for a given offset.
     *
     * @param  mixed  $offset
     * @return mixed
     */
    public function offsetGet($offset): mixed
    {
        return $this->getAttribute($offset);
    }

    /**
     * Set the value for a given offset.
     *
     * @param  mixed  $offset
     * @param  mixed  $value
     * @return null
     */
    public function offsetSet($offset, $value): void
    {
        $this->setAttribute($offset, $value);
    }

    /**
     * Unset the value for a given offset.
     *
     * @param  mixed  $offset
     * @return null
     */
    public function offsetUnset($offset): void
    {
        unset($this->attributes[$offset], $this->relations[$offset]);
    }

    /**
     * Get the format for database stored dates.
     *
     * @return string
     */
    public function getDateFormat()
    {
        if (!$this->dateFormat) {
            $connectionName = $this->getConnectionName();
            
            return AttributeCache::$dateFormatCache[$connectionName]
                ?? (AttributeCache::$dateFormatCache[$connectionName] = $this->getConnection()->getQueryGrammar()->getDateFormat());
        }
        
        return $this->dateFormat;
    }

    /**
     * Get the type of cast for a model attribute.
     *
     * @param  string  $key
     * @return string
     */
    protected function getCastType($key)
    {
        $attributeCache = $this->getFilledAttributeCache($key);
        
        if (!isset($attributeCache->castType[$key])) {
            $attributeCache->castType[$key] = trim(strtolower($attributeCache->casts[$key]));
        }
        
        return $attributeCache->castType[$key];
    }

    /**
     * Get the casts array.
     *
     * @return array
     */
    public function getCasts()
    {
        $attributeCache = $this->getAttributeCache();
        
        return $attributeCache->casts;
    }
    
    /**
     * Get the table associated with the model.
     *
     * @return string
     */
    public function getTable()
    {
        if (!isset($this->table)) {
            $attributeCache = $this->getAttributeCache();
            
            return $attributeCache->table;
        }

        return $this->table;
    }
    
    /**
     * Get the database connection for the model.
     *
     * @return \Illuminate\Database\Connection
     */
    public function getConnection()
    {
        $attributeCache = $this->getAttributeCache();
        
        $connectionName = $this->getConnectionName();
        
        if (!isset($attributeCache->connection) || $connectionName !== $attributeCache->connectionName) {
            $attributeCache->connection = static::resolveConnection($connectionName);
            $attributeCache->connectionName = $connectionName;
        }
        
        return $attributeCache->connection;
    }

}
