<?php

namespace InSegment\ApiCore\Traits;

use InSegment\ApiCore\Services\MicroEvent;

/**
 * To be used with \Illuminate\Database\Eloquent\Model
 */
trait Instantiatible
{
    /**
     * Register an instantiated model event with the dispatcher.
     *
     * @param  \Closure|string  $callback ($instantiatibleModel, array $instantiator = [
     *     'model' => \Illuminate\Database\Eloquent\Model,
     *     'function' => __FUNCTION__,
     *     ...
     * ])
     * @return void
     */
    public static function instantiated($callback)
    {
        MicroEvent::registerEvent(static::class, 'instantiated', $callback);
    }
    
    /**
     * Fire instantination event with data
     * 
     * @param array $instantiator [
     *     'model' => \Illuminate\Database\Eloquent\Model,
     *     'function' => __FUNCTION__,
     *     ...
     * ]
     * @return bool
     */
    public function fireInstantination($instantiator)
    {
        return MicroEvent::fireEvent(static::class, 'instantiated', $this, $instantiator);
    }
    
    /**
     * Fire instantiated event on newInstance
     * 
     * @param array $attributes
     * @param bool $exists
     * @return static
     */
    public function newInstance($attributes = [], $exists = false)
    {
        $newModel = parent::newInstance($attributes, $exists);
        $newModel->fireInstantination([
            'model' => $this,
            'function' => 'newInstance',
            'attributes' => $attributes,
            'exists' => $exists
        ]);
        return $newModel;
    }
    
    /**
     * Fire instantiated event on newFromBuilder
     * 
     * @param array $attributes
     * @param \Illuminate\Database\Connection|null $connection
     * @return static
     */
    public function newFromBuilder($attributes = [], $connection = null) {
        $newModel = parent::newFromBuilder($attributes, $connection);
        $newModel->fireInstantination([
            'model' => $this,
            'function' => 'newFromBuilder',
            'attributes' => $attributes,
            'connection' => $connection
        ]);
        return $newModel;
    }
    
}
