<?php

namespace InSegment\ApiCore\Traits;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use InSegment\ApiCore\Facades\DTODefs;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Services\SliceExchange;
use InSegment\ApiCore\Models\SliceOperation;
use InSegment\ApiCore\Interfaces\ApiKeyInterface;
use InSegment\ApiCore\Api\DataTransferObject;

use Symfony\Component\HttpFoundation\Response;

trait ProvidesApiEntryPoints
{
    /**
     * Authorization through ApiKey in JSON
     * 
     * If the Api Key is failed to set, the response status will be 401 Unauthorized
     * 
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function authorizeApiKey(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('sessionAuth', $apiKey);
        app('session')->put(['api_key' => $apiKey->getPublicKey()]);
        
        return $this->apiResponse();
    }
    
    /**
     * Get tables structure for slice
     * 
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function getTablesSliceStructure(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('getTablesSliceStructure', $apiKey);

        $data = app('request')->all();
        
        if (!($data['schema_name'] ?? null)) {
            throw new \Exception('Schema name of the client platform must be specified!');
        }
        
        if (isset($data['slice_exchange_structure'])) {
            $sliceExchangeStructure = $data['slice_exchange_structure'];
        } else {
            $sliceExchangeStructure = DTODefs::getSliceExchangeStructure($data['slice_exchange_name']);
        }
        
        $operation = SliceOperation::newSlice(false);
        $operation->setSchema($data['schema_name']);
        $instanceExchange = new SliceExchange($operation, $sliceExchangeStructure);
        return $this->apiResponse($instanceExchange->packForExchange());
    }

    /**
     * Get or create Transaction
     * 
     * The format of input:
     * {
     *   "root": "App\\Campaign",
     *   "count": {
     *     "campaigns": ...,
     *     "campaign_templates": ...,
     *     "campaign_template_mappings": ...,
     *     "contacts_template_mapping": ...,
     *     "companies":...,
     *     "company_ranges": ...,
     *     "company_industries": ...,
     *     "lists": ...,
     *     "contacts": ...,
     *     "contact_template_mappings": ...,
     *     "contact_titles": ...,
     *     "contact_phones": ...,
     *     "proof_links": ...
     *   }
     * }
     * 
     * @return \Illuminate\Http\Response
     */
    public function acquireTransaction(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('acquireTransaction', $apiKey);
        $data = app('request')->all();
        
        Transactor::acquire($data);
        
        return $this->apiResponse();
    }
    
    /**
     * Close Transaction
     * 
     * @return \Illuminate\Http\Response
     */
    public function closeTransaction(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('closeTransaction', $apiKey);
        $data = app('request')->all();
        $clear = data_get($data, 'clear', true);
        
        $result = Transactor::close($clear);
        
        $resultOfTransaction = DTODefs::getResultOfTransaction();
        if (isset($resultOfTransaction)) {
            $result['result'] = $resultOfTransaction;
        }

        return $this->apiResponse($result);
    }
    
    /**
     * Undo Transaction
     * 
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function showTransactionStatus(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('showTransactionStatus', $apiKey);
        $columns = data_get(app('request')->all(), 'columns');
        return $this->apiResponse(Transactor::status($columns), 'status');
    }

    /**
     * Undo Transaction
     * 
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function undoTransaction(ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('undoTransaction', $apiKey);
        Transactor::undoTransaction();

        return $this->apiResponse();
    }

    /**
     * Set/Add Object
     * 
     * @param string $type
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function set($type, ApiKeyInterface $apiKey)
    {
        return $this->import($type, $apiKey, false);
    }
    
    /**
     * Update Object
     * 
     * @param string $type
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function update($type, ApiKeyInterface $apiKey)
    {
        return $this->import($type, $apiKey, true);
    }
    
    /**
     * Set or update Object
     * 
     * @param string $type
     * @param bool $update
     * @return \Illuminate\Http\Response
     */
    private function import($type, ApiKeyInterface $apiKey, bool $update)
    {
        $this->checkApiKeyPermission('import' . ucfirst(Str::camel($type)), $apiKey);
        $data = app('request')->json()->all();
        
        $result = Transactor::import($type, $data, $update);
        
        if ($result === DataTransferObject::DTO_IMPORT_REQUIRE) {
            return $this->apiResponse(Str::singular($type), 'require');
        }
        
        return $this->apiResponse();
    }
    
    /**
     * Export an object
     * 
     * @param string $type
     * @param ApiKeyInterface $apiKey
     * @return \Illuminate\Http\Response
     */
    public function export($type, ApiKeyInterface $apiKey)
    {
        $this->checkApiKeyPermission('export' . ucfirst(Str::camel($type)), $apiKey);
        $data = app('request')->json()->all();
        
        $result = Transactor::watchingTables(function (Transactor $transactor) use (&$type, &$data) {
            return (new DataTransferObject($type, $data))->processOutgoing();
        });
        
        return $this->apiResponse($result, 'data');
    }
    
    /**
     * Return response
     *
     * @param array|string $data response data
     * @param string $root
     *
     * @return \Illuminate\Http\Response
     */
    public function apiResponse($data = null, string $root = 'data')
    {
        $responseData = ['status' => Response::HTTP_OK];
        
        if (isset($data)) {
            Arr::set($responseData, $root, $data);
        }
        
        return response()->json($responseData, Response::HTTP_OK);
    }
    
    /**
     * Check ApiKeyInterface permission if the Key supports permissions
     * 
     * @param string $permission
     * @param ApiKeyInterface $apiKey
     * @return null
     */
    public function checkApiKeyPermission(string $permission, ApiKeyInterface $apiKey)
    {
        if (method_exists($apiKey, 'authorize')) {
            $apiKey->authorize($permission);
        }
    }
}
