<?php

namespace InSegment\ApiCore\Traits;

use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

trait RunsProcessesTrait
{
    use Loggable;
    
    /**
     * Run process with some input and/or callback
     * 
     * @param \Symfony\Component\Process\Process $process
     * @param callable|null|$callback
     * @return int|null
     * @throws \Symfony\Component\Process\Exception\ProcessFailedException
     */
    protected function processWithInput($process, $callback = null)
    {
        $command = $process->getCommandLine();
        $process->run($callback);
        
        if (!$process->isSuccessful()) {
            $logger = $this->getLogger();
            $errorData = $process->getErrorOutput();
            if ($errorData) {
                if ($logger !== null) {
                    $logger->push("{$command} > ERROR", $errorData);
                } else {
                    fwrite(STDERR, $errorData);
                }
            }
            
            throw new ProcessFailedException($process);
        }
        
        return $process->getExitCode();
    }
    
    /**
     * Run process with output redirected to a resource or just return output
     * 
     * @param \Symfony\Component\Process\Process $process
     * @param resource|null $outputResource
     * @return string|int|null
     * @throws \Symfony\Component\Process\Exception\ProcessFailedException
     */
    protected function processWithOutput($process, $outputResource = null)
    {
        $command = $process->getCommandLine();
        $logger = $this->getLogger();
        if ($outputResource !== null) {
            try {
                $process->start();
                foreach ($process as $type => $data) {
                    if ($process::OUT === $type) {
                        fwrite($outputResource, $data);
                    } else if ($data) {
                        if ($logger !== null) {
                            $logger->push("{$command} > ERROR", $data);
                        } else {
                            fwrite(STDERR, $data);
                        }
                    }
                }
                
                $result = $process->getExitCode();
            } finally {
                $process->stop();
            }
        } else {
            $process->run();
            $result = $process->getOutput();
            $errorData = $process->getErrorOutput();
            if ($errorData) {
                if ($logger !== null) {
                    $logger->push("{$command} > ERROR", $errorData);
                } else {
                    fwrite(STDERR, $errorData);
                }
            }
        }
        
        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }
        
        return $result;
    }
    
    /**
     * Create process for command
     * 
     * @param string $commandName
     * @param array $arguments
     * @return \Symfony\Component\Process\Process
     */
    protected function getProcess(string $commandName, array $arguments)
    {
        $isWindows = stripos(PHP_OS, 'WIN') === 0;
        $quotedArgs = [];
        
        foreach ($arguments as $argName => $argValue) {
            if ($isWindows) {
                $quoteValue = '"'. addcslashes($argValue, '"') . '"';
            } else {
                $quoteValue = "'". addcslashes($argValue, "'") . "'";
            }
            
            if (is_int($argName)) {
                $quotedArgs[] = $quoteValue;
            } else {
                $quotedArgs[] = "{$argName}{$quoteValue}";
            }
        }
        
        $implodeQuotedArgs = implode(' ', $quotedArgs);
        $commandLine = "{$commandName} {$implodeQuotedArgs}";

        //TODO temp solution for the backward compatibility.
        $process = (method_exists(Process::class, 'fromShellCommandline'))
            ? Process::fromShellCommandline($commandLine)
            : new Process($commandLine);

        $process->setTimeout(null);
        return $process;
    }
    
}
