<?php

namespace InSegment\ApiCore\Traits;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

use InSegment\ApiCore\Services\DiffSlice;
use InSegment\ApiCore\Services\Transactor;
use InSegment\ApiCore\Services\JoinResolver;
use InSegment\ApiCore\Services\BufferedBuilder;
use InSegment\ApiCore\Scopes\TransactionSearchScope;

use InSegment\ApiCore\Services\MicroEvent;

/**
 * To be used with \Illuminate\Database\Eloquent\Model
 */
trait Transactible
{
    use Instantiatible;
    use EventsReplace;
    use CachesAttributes;
    
    /**
     * Custom inverse relations by JoinResolver
     * 
     * @see \InSegment\ApiCore\Services\JoinResolver
     * @var mixed
     */
    private $inverseRelations;
    
    /**
     * Whether the Model was already appointed to transaction
     * 
     * @var bool 
     */
    private $transactionAppointed;
    
    /**
     * The Model is fully prepared for transactor and should be saved directly upon call save
     * 
     * @var bool 
     */
    private $directlySave;
    
    /**
     * Really is incrementing
     * 
     * @var bool 
     */
    public $_incrementing;
    
    /**
     * Boot method
     * 
     * @return null
     */
    public static function bootTransactible()
    {
        static::instantiated(function ($model, $instantiator) {
            switch ($instantiator['function']) {
                
                /**
                 * This is a bit tricky. A Relation has internal related Model instance,
                 * which is not an actual Model until it comes from Builder. It is possible
                 * to set something on that internal instance through ::macro but anything
                 * custom that was set is lost once the Model is loaded from Builder.
                 * 
                 * Yet, the new instance from Builder gets created through the method newInstance
                 * on the original internal instance - so overriding it, is possible to copy
                 * some custom values to the new Model
                 */
                case 'newInstance':
                    if (isset($instantiator['model']->inverseRelations)) {
                        $model->inverseRelations = $instantiator['model']->inverseRelations;

                        if (!$instantiator['exists']) {
                            JoinResolver::apply($instantiator['model'], $model);
                        }
                    }
                break;
                
                /**
                 * When a model comes from Builder, JoinRelover can be applied to restore inverseOf relation
                 */
                case 'newFromBuilder':
                    if (isset($instantiator['model']->inverseRelations)) {
                        JoinResolver::apply($instantiator['model'], $model);
                    }
                break;
            }
        });
    }
    
    /**
     * Appoint Model transaction
     * 
     * Is is used as first step of import, the key of the Model is set there.
     * It is the internal id key, so it is always primary and auto increment,  but we have two 'temporary' tables
     * we can write the model to: insert table and update table, and update one does not auto increment.
     * That is okay. Also, the keys are preallocated. So we have knowledge of what exactly we'll have as a primary key
     * before the record is written to the actual table.
     * 
     * This should be used before creating related Models, if any, as the modification methods
     * for Models with  appointed transactions may be using collecting scenario of write.
     * In this case, there will be bulk queries using INSERT IGNORE, REPLACE or  INSERT ... ON DUPLICATE KEY UPDATE
     * (handled by another abstraction)
     * 
     * @return $this
     */
    public function appointTransactionWrite()
    {
        $this->setKeyByAppointment()->setTablesForSaveByAppointment();
        
        $this->transactionAppointed = true;
        
        return $this;
    }
    
    /**
     * Receive Model in transaction stating that it is actually written
     * 
     * @param bool $wasDirty
     * @param bool $dontSave
     * @return $this
     */
    public function performTransactionReceive(bool $wasDirty, bool $dontSave)
    {
        Transactor::receive($wasDirty, $dontSave, $this);
        
        return $this;
    }
    
    /**
     * Save the model to the database.
     * 
     * If the save is performed within a transaction, write Model to one of the transaction tables instead of main one;
     * The following description applied to what happens when the Model is saved that way:
     * 
     * Model can exist in original table, in insert table or in update table. If the table exists in one of these,
     * it automatically exists in union of three tables (update + insert + original, in order of preference),
     * and thus have its $model->exists set as true. If a Model exists in union, but not in original table, we cannot
     * "INSERT" it again into that spot - we need to know that we need to "UPDATE" it with recent data. Eloquent
     * will automatically set "INSERT" or "UPDATE" statement depending on whether $model->exists - so we need
     * to unset it, if we need insert, or set if we need an update.
     * 
     * Business logic of a model can depend on its existence, yet if the $model->exists was set to true to force
     * "UPDATE" statement, the logic can falsely detect that Model "existed" in the original table. That is why
     * we have 'importing' event which is fired there as a replacement for 'saving' event to know for certain,
     * whether the Model is truly new for us. However, that event will be fired event if the model will not be saved.
     * 
     * If the Model is changed (dirty), it will be saved. But before the save actually occurs, we need one final blow
     * of relating from the parent Model to current: if we have such a relation, it will force equality constraints
     * to be set as attributes on the Model.
     * 
     * Finally, the Transactor receives the Model.
     *
     * @param array $options
     * @param bool $direct
     * @param \Illuminate\Database\Eloquent\Relations\Relation|null $relationToModel
     * @return bool
     */
    public function save(array $options = [], bool $direct = false, $relationToModel = null)
    {
        if (!$this->directlySave && !$direct && Transactor::isWatching()) {
            if (!$this->transactionAppointed) {
                $this->appointTransactionWrite();
            }
            
            // allow saving-line events using original existence
            $doNotSave = MicroEvent::fireEvent(static::class, 'importing', $this) === false;
            if (!$doNotSave) {
                // if the model is not dirty, it can have no key, it idicates that it was recieved, but not written, because it matches defaults
                $wasDirty = $this->isDirty();
                // $wasExists = $this->exists;
                $keyName = $this->getKeyName();
                $key = $this->getKey();

                if ($wasDirty || $key !== null) {
                    $this->exists = Transactor::getExistence(static::class, $key, $keyName);
                }

                // do not write, but receive unchanged models
                if ($wasDirty) {
                    if (isset($relationToModel)) {
                        $this->directlySave = true;
                        $this->attributes = $relationToModel->constraintsToAttributes($this);
                        
                        if ($relationToModel instanceof BelongsTo) {
                            $return = parent::save($options);
                        } else {
                            $return = $relationToModel->save($this);
                        }
                        
                        $this->directlySave = false;
                        $this->transactionAppointed = false;
                    } else {
                        $return = parent::save($options);
                        $this->transactionAppointed = false;
                    }
                }
            }
            
            // even if the event tells us to ignore the Model, the counters should be increased
            // so we know, that it is not forgotten to be sent in transaction
            $this->performTransactionReceive($wasDirty ?? false, $doNotSave);
            return $return ?? false;
        } else {
            return parent::save($options);
        }
    }
    
    /**
     * Delete the model from the database.
     *
     * @throws \Exception
     * @return bool|null
     */
    public function delete()
    {
        if (Transactor::isWatching()) {
            $allowed = Transactor::getDeletionEnabledTables();
            // do not use $this->getTable(), as it might already be transaction insert/update table
            $keyName = $this->getKeyName();
            $keyVal = $this->getKey();
            $currentTable = $this->getTable();
            $origTable = (new static)->getTable();
            $deleteFromTable = $currentTable !== $origTable
                ? (Transactor::getExistence(static::class, $keyVal, $keyName)
                    ? $currentTable : $origTable
                ) : $origTable;
            
            if (!isset($allowed[$origTable])) {
                $class = static::class;
                throw new \Exception("Deleting of model of class '{$class}' is currently not allowed through Transactor!");
            } else {
                Transactor::markDeleted($deleteFromTable, $keyName, $keyVal);
            }
        } else {
            return parent::delete();
        }
    }
    
    /**
     * Get the primary key for a new model from the Transactor, if it is new
     * 
     * @return $this
     */
    private function setKeyByAppointment()
    {
        if ($this->_incrementing && !$this->exists) {
            $this->setAttribute($this->getKeyName(), Transactor::nextId(static::class));
        }
        
        return $this;
    }
    
    /**
     * Updating tables, as we must separate new Models into 'insert' and 'update' transaction tables
     * and also prevent writing to the original table
     * 
     * @return $this
     */
    private function setTablesForSaveByAppointment()
    {
        $transactionInsertTable = Transactor::getInsertTable(static::class, true);
        
        if ($this->exists) {
            $this->setTable(Transactor::getUpdateTable(static::class, true));
        } else {
            $this->setTable($transactionInsertTable);
        }
        
        return $this;
    }
    
    /**
     * Create a new Eloquent query builder for the model.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \InSegment\ApiCore\Services\BufferedBuilder|static
     */
    public function newEloquentBuilder($query)
    {
        return new BufferedBuilder($query);
    }
    
    /**
     * Access to $this->inverseRelations with JoinResolver
     * 
     * @return mixed
     */
    public function &accessInverseRelations()
    {
        return $this->inverseRelations;
    }
    
    /**
     * First, check if the model needs to be booted and if so, do it.
     * Eloquent calls this method upon any construction of new Model
     * 
     * Then, if the Model has incrementing key, store original value of _incrementing for service use
     * and disguise as non-incrementing, and having string key type
     * 
     * @return mixed 
     */
    protected function bootIfNotBooted()
    {
        $ret = parent::bootIfNotBooted();
        
        $this->_incrementing = $this->incrementing;
        
        if (TransactionSearchScope::isEnabled(static::class) && $this->incrementing) {
            // prevent Eloquent from trying to set primary key to the last inserted row
            $this->incrementing = false;
            $this->keyType = 'string';
            $this->attributeCacheKey = 'transactionScope';
        }
        
        return $ret;
    }
    
    /**
     * Register an importing model event with the dispatcher.
     *
     * @param  \Closure|string  $callback
     * @return void
     */
    public static function importing($callback)
    {
        MicroEvent::registerEvent(static::class, 'importing', $callback);
    }

    /**
     * Register an importMade model event with the dispatcher.
     *
     * @param  \Closure|string  $callback
     * @return void
     */
    public static function importMade($callback)
    {
        MicroEvent::registerEvent(static::class, 'importMade', $callback);
    }

    /**
     * Method that is called upon the Model is loaded or being created on import
     * The returned result of the method will be set ad Model being imported, and
     * if return value evaluates to false (false, null, 0 and so on) the Model won't be imported
     * 
     * @param mixed $input
     * @param mixed $context
     * @param bool $isUpdate
     * @return $this
     */
    public function newFromImport(&$input, &$context, $isUpdate)
    {
        $this->fireInstantination([
            'model' => $this,
            'function' => 'newFromImport',
            'input' => &$input,
            'context' => &$context,
            'isUpdate' => $isUpdate
        ]);
        
        if (MicroEvent::fireEvent(static::class, 'importMade', $this) === false) {
            return;
        }

        return $this->appointTransactionWrite();
    }
    
    /**
     * Set the specific relationship in the model.
     *
     * @param  string  $relation
     * @param  mixed  $value
     * @return $this
     */
    public function setRelation($relation, $value)
    {
        if (JoinResolver::$enabled) {
            if ($value instanceof Model) {
                if ($value->inverseRelations) {
                    JoinResolver::applyEager($value, $this);
                }
            } else if ($value instanceof Collection) {
                foreach ($value as $model) {
                    if ($model->inverseRelations) {
                        JoinResolver::applyEager($model, $this);
                    }
                }
            }
        }
        
        $this->relations[$relation] = $value;

        return $this;
    }
    
}
